package ru.yandex.direct.model.generator.old.conf;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import com.typesafe.config.Config;

import ru.yandex.direct.model.generator.old.javafile.Util;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.model.generator.old.conf.RelationshipConf.SourceProperty.ID_FIELD;
import static ru.yandex.direct.model.generator.old.conf.RelationshipConf.SourceProperty.ID_TYPE;
import static ru.yandex.direct.model.generator.old.conf.RelationshipConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.RelationshipConf.SourceProperty.PACKAGE;

/**
 * Описание модели отношений между двумя другими.
 * <p>
 * Может быть создано из поля атрибута Relationship — Задает связь через поле, на котором обозначена, между классом,
 * содержащим поле, и указанным классом, на который это поле указывает как внешний ключ.
 * Например, Banner.adGroupId -> AdGroup
 * <p>
 * Также может быть создана из отдельного конфига, в таком случае дочерний и родительский классы, а также имя ключа
 * и его тип нужно указать вручную.
 */
@ParametersAreNonnullByDefault
public class RelationshipConf implements ModelConf, UpperLevelModelConf {
    private final String name;
    private final ClassName parent;
    private final String packageName;
    private final ClassName className;
    private final ClassName childClass;
    private final TypeName parentIdType;
    private final String parentIdName;
    private final String sourceFile;

    private RelationshipConf(String name, ClassName parent, @Nullable String packageName,
                             ClassName childClass, TypeName parentIdType, String parentIdName,
                             @Nullable String sourceFile) {
        this.name = checkNotNull(name);
        this.parent = parent;
        this.packageName = packageName;
        this.className = ClassName.get(packageName, name);
        this.childClass = childClass;
        this.parentIdType = parentIdType;
        this.parentIdName = parentIdName;
        this.sourceFile = sourceFile;
    }

    public static RelationshipConf fromConfig(Config config, @Nullable String defaultPackage,
                                              ClassName childClass, TypeName parentIdType, String parentIdName,
                                              @Nullable String sourceFile) {
        String name = config.hasPath(NAME) ? config.getString(NAME) : "";
        return of(name, config.getString(SourceProperty.PARENT), defaultPackage, childClass,
                parentIdType, parentIdName, sourceFile);
    }

    public static RelationshipConf fromConfig(Config config, String sourceFileName) {
        String defaultPackage = config.getString(PACKAGE);
        String name = config.getString(NAME);
        String idType = config.hasPath(ID_TYPE) ? config.getString(ID_TYPE) : "Long";

        return new RelationshipConf(name,
                Util.classNameOf(config.getString(SourceProperty.PARENT), defaultPackage),
                defaultPackage,
                Util.classNameOf(config.getString(SourceProperty.CHILD), defaultPackage),
                Util.typeNameOf(idType, defaultPackage),
                config.getString(ID_FIELD),
                sourceFileName
        );
    }

    public static RelationshipConf of(String name, String parent, @Nullable String defaultPackage,
                                      ClassName childClass, TypeName parentIdType, String parentIdName,
                                      @Nullable String sourceFile) {
        return new RelationshipConf(name, Util.classNameOf(parent, defaultPackage), defaultPackage,
                childClass, parentIdType, parentIdName,
                sourceFile);
    }

    @Override
    public List<String> getNestedModelsAttributesFullPaths() {
        return List.of();
    }

    @Override
    public List<AttrConf> getAttrs() {
        return List.of();
    }

    @Override
    public List<String> getAttributesFullPaths() {
        return List.of();
    }

    @Override
    public List<ModelConf> getAllModelConfs() {
        return List.of(this);
    }

    @Override
    public Type getType() {
        return Type.RELATIONSHIP_MODEL;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getPackageName() {
        return packageName;
    }

    @Override
    public String getComment() {
        return "";
    }

    @Override
    public ClassName getClassName() {
        return className;
    }

    @Override
    public List<AnnotationConf> getAnnotations() {
        return List.of();
    }

    @Override
    public List<String> getAttrNames() {
        return List.of();
    }

    @Override
    public List<String> getSupersFullNames() {
        return List.of();
    }

    @Override
    public List<String> getExtendsList() {
        return List.of();
    }

    @Override
    public List<String> getExtendsAndImplementsFullNames() {
        return List.of();
    }

    public ClassName getParent() {
        return parent;
    }

    public ClassName getChildClass() {
        return childClass;
    }

    public TypeName getParentIdType() {
        return parentIdType;
    }

    public String getParentIdName() {
        return parentIdName;
    }

    @Nullable
    @Override
    public String getSourceFile() {
        return sourceFile;
    }

    @Override
    public String getFullName() {
        return className.toString();
    }

    @Override
    public String toString() {
        return "RelationshipConf{" +
                "name='" + name + '\'' +
                ", parent=" + parent +
                ", packageName='" + packageName + '\'' +
                ", className=" + className +
                ", childClass=" + childClass +
                ", parentIdType=" + parentIdType +
                ", parentIdName='" + parentIdName + '\'' +
                ", sourceFile='" + sourceFile + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RelationshipConf that = (RelationshipConf) o;
        return Objects.equals(name, that.name) &&
                Objects.equals(parent, that.parent) &&
                Objects.equals(packageName, that.packageName) &&
                Objects.equals(className, that.className) &&
                Objects.equals(childClass, that.childClass) &&
                Objects.equals(parentIdType, that.parentIdType) &&
                Objects.equals(parentIdName, that.parentIdName) &&
                Objects.equals(sourceFile, that.sourceFile);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, parent, packageName, className, childClass, parentIdType, parentIdName, sourceFile);
    }

    static class SourceProperty {
        static final String PACKAGE = "package";
        static final String NAME = "name";
        static final String PARENT = "parent";
        static final String CHILD = "child";
        static final String ID_FIELD = "parentIdField";
        static final String ID_TYPE = "parentIdType";
    }
}
