package ru.yandex.direct.model.generator.old.javafile;

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.lang.model.element.Modifier;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.Copyable;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.model.generator.old.conf.AnnotationConf;
import ru.yandex.direct.model.generator.old.conf.AttrConf;
import ru.yandex.direct.model.generator.old.spec.ClassSpec;

import static java.lang.String.format;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.model.generator.old.javafile.Util.makeJsonSubtypes;

class ModelClassBuilder {

    static final String ID_FIELD = "id";
    static final ClassName MODEL = ClassName.get(Model.class);
    static final ClassName MODEL_WITH_ID = ClassName.get(ModelWithId.class);
    static final ClassName MODEL_PROPERTY = ClassName.get(ModelProperty.class);
    static final Function<AttrConf, String> ATTR_TO_MODEL_PROPERTY_NAME = attr -> Util.upperUnderscore(attr.getName());
    static final ClassName COPYABLE_MODEL = ClassName.get(Copyable.class);
    static final String COPY_METHOD_NAME = "copy";
    static final String COPY_TO_METHOD_NAME = "copyTo";
    private static final String TARGET_OBJECT_NAME = "target";

    static final String ALL_MODEL_PROPERTIES_METHOD_NAME = "allModelProperties";
    static final Type MODEL_PROPERTY_TYPE = new TypeReference<ModelProperty<?, ?>>() {
    }.getType();
    static final ParameterizedTypeName RETURN_OF_ALL_MODEL_PROPERTIES_METHOD =
            ParameterizedTypeName.get(Set.class, MODEL_PROPERTY_TYPE);
    static final ClassName IMMUTABLE_SET = ClassName.get(ImmutableSet.class);

    private final ClassSpec spec;

    private final Map<AttrConf, MethodSpec> getters = new HashMap<>();
    private final Map<AttrConf, MethodSpec> setters = new HashMap<>();

    ModelClassBuilder(ClassSpec spec) {
        this.spec = spec;
    }

    TypeSpec build() {
        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(spec.getClassName())
                .addAnnotation(Util.createGeneratedAnnotation(spec))
                .addJavadoc(spec.getComment() != null ? spec.getComment() : "");

        for (AnnotationConf annotationConf : spec.getAnnotations()) {
            classBuilder.addAnnotation(annotationConf.toSpec());
        }

        if (spec.getModifiers().isEmpty()) {
            classBuilder.addModifiers(Modifier.PUBLIC);
        } else {
            spec.getModifiers().forEach(classBuilder::addModifiers);
        }

        if (!spec.getJsonSubtypes().isEmpty()) {
            classBuilder.addAnnotation(makeJsonSubtypes(spec.getJsonSubtypes(), spec.isJsonSubtypesWithNameValue()));
        }

        if (spec.getExtendsClassName() != null) {
            classBuilder.superclass(spec.getExtendsClassName());
        }

        boolean onlyPropHolders = spec.getSuperInterfaces()
                .stream()
                .allMatch(i -> i.toString().endsWith("PropHolder"));

        if (onlyPropHolders && spec.isGenerateGetPropsMethod()) {
            classBuilder.addSuperinterface(hasInterfaceIdField() ? MODEL_WITH_ID : MODEL);
        } else {
            classBuilder.addSuperinterfaces(spec.getSuperInterfaces());
        }

        spec.getInheritedAttributes()
                .forEach(attr -> classBuilder.addMethod(makeOverridingWithSetter(attr)));

        for (AttrConf attr : spec.getAttrs()) {
            FieldSpec field = makeField(attr);
            MethodSpec getter = getterFor(attr);
            MethodSpec setter = setterFor(attr);
            MethodSpec withSetter = makeWithSetter(attr, setter);

            if (field != null) {
                classBuilder
                        .addField(field);
            }

            if (onlyPropHolders && spec.isGenerateGetPropsMethod()) {
                classBuilder.addField(makeProp(attr));
            }

            classBuilder
                    .addMethod(getter)
                    .addMethod(setter)
                    .addMethod(withSetter);
        }

        if (spec.isGenerateGetPropsMethod()) {
            classBuilder.addMethod(makeGetModelPropertiesMethod());
        }

        if (spec.isGenerateCopyMethod()) {
            if (!spec.getExtends().isEmpty() && !spec.isSuperHasCopyMethod()) {
                throw new IllegalStateException(
                        String.format("Can't generate copy method for %s. Super class must have copy method",
                                spec.getClassName().simpleName()));
            }

            if (!spec.isSuperHasCopyMethod()) {
                classBuilder.addSuperinterface(COPYABLE_MODEL);
            }

            classBuilder.addMethod(makeCopyToMethod());

            if (!spec.getModifiers().contains(Modifier.ABSTRACT)) {
                classBuilder.addMethod(makeCopyMethod());
            }
        }

        classBuilder
                .addMethod(makeToString())
                .addMethod(makeEquals())
                .addMethod(makeHashCode());

        return classBuilder.build();
    }

    private MethodSpec makeToString() {
        MethodSpec.Builder builder = MethodSpec.methodBuilder("toString")
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(Override.class)
                .returns(String.class)
                .addStatement("$1T sb = new $1T($2S)", StringBuilder.class, spec.getClassName() + "{");

        boolean needLeadingComma = false;

        if (!spec.getExtends().isEmpty()) {
            builder.addStatement("sb.append(\"super=\").append(super.toString())");
            needLeadingComma = true;
        }

        List<AttrConf> attrs = spec.getAttrs();
        for (AttrConf attr : attrs) {
            if (isAlias(attr)) {
                continue;
            }
            if (needLeadingComma) {
                builder.addStatement("sb.append(\", $1L=\").append($1N)", attr.getName());
            } else {
                builder.addStatement("sb.append(\"$1L=\").append($1N)", attr.getName());
            }
            needLeadingComma = true;
        }

        builder.addStatement("sb.append('}')");
        builder.addStatement("return sb.toString()");
        return builder.build();
    }

    private MethodSpec makeEquals() {
        MethodSpec.Builder builder = MethodSpec.methodBuilder("equals")
                .addParameter(Object.class, "o")
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(Override.class)
                .returns(TypeName.BOOLEAN)
                .beginControlFlow("if (this == o)")
                .addStatement("return true")
                .endControlFlow()
                .beginControlFlow("if (o == null || getClass() != o.getClass())")
                .addStatement("return false")
                .endControlFlow()
                .addStatement("$1T that = ($1T) o", spec.getClassName());

        if (!spec.getExtends().isEmpty()) {
            builder.addStatement("if (!super.equals(o)) \n return false");
        }

        if (spec.getAttrs().isEmpty()) {
            builder.addStatement("return true");
        } else {
            String eqs =
                    spec.getAttrs().stream()
                            .filter(a -> !isAlias(a))
                            .map(a -> format("$1T.equals(%s, that.%s)", a.getName(), a.getName()))
                            .collect(joining("\n && "));

            builder.addStatement("return " + eqs, Objects.class);
        }

        return builder.build();
    }

    private MethodSpec makeHashCode() {
        List<String> fields = spec.getAttrs().stream()
                .filter(a -> !isAlias(a))
                .map(AttrConf::getName)
                .collect(toList());

        if (!spec.getExtends().isEmpty()) {
            fields.add(0, "super.hashCode()");
        }

        MethodSpec.Builder builder = MethodSpec.methodBuilder("hashCode")
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(Override.class)
                .returns(TypeName.INT)
                .addStatement("return $T.hash(\n$L\n)",
                        Objects.class,
                        String.join(",\n", fields)
                );
        return builder.build();
    }

    private FieldSpec makeField(AttrConf attr) {
        if (isAlias(attr)) {
            return null;
        }
        FieldSpec.Builder builder = FieldSpec.builder(attr.getType(), attr.getName(), Modifier.PRIVATE);

        for (AnnotationConf annotationConf : attr.getAnnotations()) {
            if (!annotationConf.annotateField()) {
                continue;
            }
            builder.addAnnotation(annotationConf.toSpec());
        }

        return builder.build();
    }

    private FieldSpec makeProp(AttrConf attr) {
        final CodeBlock initializer = CodeBlock.of("\n $1T.create($2N.class, $3S, $2N::$4N, $2N::$5N)",
                MODEL_PROPERTY, spec.getName(),
                attr.getName(), getterFor(attr), setterFor(attr));

        TypeName typeArg = attr.getType().isPrimitive() ? attr.getType().box() : attr.getType();
        return FieldSpec.builder(
                ParameterizedTypeName.get(MODEL_PROPERTY, spec.getClassName(), typeArg),
                ATTR_TO_MODEL_PROPERTY_NAME.apply(attr),
                Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
                .addJavadoc(attr.getComment() != null ? attr.getComment() : "")
                .initializer(initializer)
                .build();
    }

    private MethodSpec makeOverridingWithSetter(AttrConf attr) {
        MethodSpec setter = setterFor(attr);
        return makeBasicWithSetter(attr, setter).addAnnotation(Override.class).build();
    }

    private MethodSpec makeWithSetter(AttrConf attr, MethodSpec setter) {
        return makeBasicWithSetter(attr, setter).build();
    }

    private MethodSpec.Builder makeBasicWithSetter(AttrConf attr, MethodSpec setter) {
        return MethodSpec.methodBuilder("with" + Util.upperCamel(attr.getName()))
                .addModifiers(Modifier.PUBLIC)
                .addParameters(setter.parameters)
                .returns(spec.getClassName())
                .addStatement("$N($L)", setter, setter.parameters.get(0).name)
                .addStatement("return this");
    }

    private MethodSpec setterFor(AttrConf attr) {
        return setters.computeIfAbsent(attr, this::makeSetter);
    }

    private MethodSpec makeSetter(AttrConf attr) {
        MethodSpec.Builder setterBuilder = MethodSpec.methodBuilder("set" + Util.upperCamel(attr.getName()))
                .addModifiers(Modifier.PUBLIC)
                .returns(void.class)
                .addStatement("this.$1L = $1L", realFieldName(attr));

        ParameterSpec.Builder parameterBuilder = ParameterSpec.builder(attr.getType(), realFieldName(attr));
        attr.getAnnotations().stream()
                .filter(AnnotationConf::annotateSetterParameter)
                .forEach(annotationConf -> parameterBuilder.addAnnotation(annotationConf.toSpec()));

        return setterBuilder
                .addParameter(parameterBuilder.build())
                .build();
    }

    private MethodSpec getterFor(AttrConf attr) {
        return getters.computeIfAbsent(attr, this::makeGetter);
    }

    private MethodSpec makeGetter(AttrConf attr) {
        MethodSpec.Builder getterBuilder = MethodSpec.methodBuilder("get" + Util.upperCamel(attr.getName()))
                .addJavadoc(attr.getComment())
                .addModifiers(Modifier.PUBLIC)
                .returns(attr.getType())
                .addStatement("return $L", realFieldName(attr));
        if (spec.isGenerateProperties() && ID_FIELD.equals(attr.getName())) {
            getterBuilder.addAnnotation(Override.class);
        }

        attr.getAnnotations().stream()
                .filter(AnnotationConf::annotateGetter)
                .forEach(annotationConf -> getterBuilder.addAnnotation(annotationConf.toSpec()));

        return getterBuilder.build();
    }

    private MethodSpec makeGetModelPropertiesMethod() {
        List<String> fields = StreamEx.of(spec.getInheritedAttributes())
                .append(spec.getAttrs())
                .remove(this::isAlias)
                .map(ATTR_TO_MODEL_PROPERTY_NAME)
                .toList();

        MethodSpec.Builder builder = MethodSpec.methodBuilder(ALL_MODEL_PROPERTIES_METHOD_NAME)
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(RETURN_OF_ALL_MODEL_PROPERTIES_METHOD)
                .addStatement("return $T.of(\n$L\n)",
                        IMMUTABLE_SET,
                        String.join(",\n", fields)
                );
        return builder.build();
    }

    private MethodSpec makeCopyToMethod() {
        MethodSpec.Builder builder = MethodSpec.methodBuilder(COPY_TO_METHOD_NAME)
                .addModifiers(Modifier.PROTECTED)
                .returns(spec.getClassName())
                .addParameter(spec.getClassName(), TARGET_OBJECT_NAME, Modifier.FINAL);

        if (spec.isSuperHasCopyMethod()) {
            builder.addStatement("super.$1L($2L)", COPY_TO_METHOD_NAME, TARGET_OBJECT_NAME);
        }

        spec.getAttrs().stream()
                .filter(a -> !isAlias(a))
                .map(AttrConf::getName)
                .forEach(atr -> builder.addStatement("$1L.$2L = this.$2L", TARGET_OBJECT_NAME, atr));

        return builder
                .addStatement("return $1L", TARGET_OBJECT_NAME)
                .build();
    }

    private MethodSpec makeCopyMethod() {
        MethodSpec.Builder builder = MethodSpec.methodBuilder(COPY_METHOD_NAME)
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc("Create field-for-field copy of this object.\n")
                .addJavadoc("<b>Not deep copy</b>\n\n")
                .addJavadoc("@see #$1L", COPY_TO_METHOD_NAME)
                .returns(spec.getClassName())
                .addStatement("final $1T $2L = new $1T()", spec.getClassName(), TARGET_OBJECT_NAME)
                .addStatement("return this.$1L($2L)", COPY_TO_METHOD_NAME, TARGET_OBJECT_NAME);

        return builder.build();
    }

    private boolean isAlias(AttrConf attr) {
        return !realFieldName(attr).equals(attr.getName());
    }

    /**
     * имеет ли интерфейс поле id типа Long среди собственных и отнаследованных аттрибутов
     */
    private boolean hasInterfaceIdField() {
        return StreamEx.of(spec.getInheritedAttributes())
                .append(spec.getAttrs().stream())
                .anyMatch(attr -> attr.getName().equals(ID_FIELD) && attr.getType().equals(TypeName.get(Long.class)));
    }

    private String realFieldName(AttrConf attr) {
        return Strings.isNullOrEmpty(attr.getAliasTo()) ? attr.getName() : attr.getAliasTo();
    }
}
