package ru.yandex.direct.model.generator.old.javafile;

import java.util.Arrays;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.lang.model.element.Modifier;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import ru.yandex.direct.model.generator.old.conf.EnumValueConf;
import ru.yandex.direct.model.generator.old.spec.EnumSpec;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.model.generator.old.javafile.Util.typeNameOf;
import static ru.yandex.direct.model.generator.old.javafile.Util.upperCamel;

class ModelEnumBuilder {
    private final EnumSpec spec;
    private final ClassName enumName;

    ModelEnumBuilder(EnumSpec spec) {
        this.spec = spec;
        this.enumName = Util.classNameOf(this.spec.getName(), spec.getPackageName());
    }

    TypeSpec build() {
        TypeSpec.Builder enumBuilder = TypeSpec.enumBuilder(enumName)
                .addAnnotation(Util.createGeneratedAnnotation(spec))
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc(spec.getComment() != null ? spec.getComment() : "Auto-generated enum");

        if (!spec.getValuesSource().isEmpty()) {
            addImportedValues(enumBuilder);
        } else {
            addConstantValues(enumBuilder);

            if (!spec.getValuesType().isEmpty()) {
                addTypedValues(enumBuilder);
            }
        }

        return enumBuilder.build();
    }

    private void addImportedValues(TypeSpec.Builder enumBuilder) {
        ClassName sourceClassName = Util.classNameOf(spec.getValuesSource(), spec.getPackageName());

        Class<?> sourceClass;
        try {
            sourceClass = ModelEnumBuilder.class.getClassLoader().loadClass(sourceClassName.reflectionName());
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("Class " + sourceClassName + " is not accessible", e);
        }

        checkState(sourceClass.isEnum(), "Class %s is not enum", sourceClassName);

        MethodSpec.Builder fromSourceBuilder = MethodSpec.methodBuilder("fromSource")
                .addParameter(sourceClassName, "value")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addAnnotation(Nullable.class)
                .returns(enumName)
                .beginControlFlow("if (value == null)")
                .addStatement("return null")
                .endControlFlow()
                .beginControlFlow("switch(value)");

        MethodSpec.Builder toSourceBuilder = MethodSpec.methodBuilder("toSource")
                .addParameter(enumName, "value")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addAnnotation(Nullable.class)
                .returns(sourceClassName)
                .beginControlFlow("if (value == null)")
                .addStatement("return null")
                .endControlFlow()
                .beginControlFlow("switch(value)");

        Object[] enumConstants;
        try {
            enumConstants = sourceClass.getEnumConstants();
        } catch (NoClassDefFoundError e) {
            throw new IllegalStateException("Could not load enumeration of " + sourceClassName, e);
        }
        for (Object obj : enumConstants) {
            @SuppressWarnings("unchecked")
            String src = ((Enum) obj).name();
            String val = src.toUpperCase();

            enumBuilder.addEnumConstant(val);
            fromSourceBuilder.addStatement("case $N:\nreturn $N", src, val);
            toSourceBuilder.addStatement("case $N:\nreturn $T.$N", val, sourceClassName, src);
        }

        fromSourceBuilder
                .addStatement("default: throw new $T($S + value)", IllegalStateException.class, "No such value: ")
                .endControlFlow();

        toSourceBuilder
                .addStatement("default: throw new $T($S + value)", IllegalStateException.class, "No such value: ")
                .endControlFlow();

        enumBuilder.addMethod(fromSourceBuilder.build())
                .addMethod(toSourceBuilder.build());
    }

    private void addConstantValues(TypeSpec.Builder enumBuilder) {
        boolean hasValuesType = !spec.getValuesType().isEmpty();

        for (EnumValueConf valueConf : spec.getValues()) {
            boolean hasTypedValue = !valueConf.getTypedValue().isEmpty();
            if (hasValuesType != hasTypedValue) {
                throw new IllegalStateException("If valuesType or typedValue of enum constant are set"
                        + " you need to set typedValue for each enum value and valuesType to enum constant both.");
            }

            TypeSpec.Builder typeSpecBuilder;
            if (hasTypedValue) {
                String typeArgumentsFormat = "String".equals(spec.getValuesType()) ? "$S" : "$L";

                typeSpecBuilder = TypeSpec.anonymousClassBuilder(typeArgumentsFormat, valueConf.getTypedValue());
            } else {
                typeSpecBuilder = TypeSpec.anonymousClassBuilder("");
            }

            if (valueConf.getComment() != null) {
                typeSpecBuilder.addJavadoc(valueConf.getComment());
            }

            if (!valueConf.getJsonProperty().isEmpty()) {
                typeSpecBuilder.addAnnotation(AnnotationSpec.builder(JsonProperty.class)
                        .addMember("value", "$S", valueConf.getJsonProperty())
                        .build());
            }

            enumBuilder.addEnumConstant(valueConf.getValue(), typeSpecBuilder.build());
        }
    }

    private void addTypedValues(TypeSpec.Builder enumBuilder) {
        String typedValueName = "typedValue";
        String typedValueGetterName = "get" + upperCamel(typedValueName);
        String enumMapName = "ENUMS";

        TypeName typedValuesType = typeNameOf(spec.getValuesType(), "");
        TypeName enumMapType = typeNameOf("Map<" + typedValuesType + ", " + enumName + ">", "");

        FieldSpec enumMapField = FieldSpec.builder(enumMapType, enumMapName)
                .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .initializer("$T.stream(values())$W.collect($T.toMap($T::$L, e -> e))",
                        Arrays.class, Collectors.class, enumName, typedValueGetterName)
                .build();

        MethodSpec constructor = MethodSpec.constructorBuilder()
                .addParameter(typedValuesType, typedValueName)
                .addStatement("this.$N = $N", typedValueName, typedValueName)
                .build();

        MethodSpec typedValueGetter = MethodSpec.methodBuilder(typedValueGetterName)
                .addModifiers(Modifier.PUBLIC)
                .returns(typedValuesType)
                .addStatement("return $L", typedValueName)
                .build();

        ParameterSpec typedValueParameter = ParameterSpec.builder(typedValuesType, typedValueName)
                .addAnnotation(Nullable.class)
                .build();

        String resultValueName = "result";
        MethodSpec fromTypedValueConverterMethod = MethodSpec.methodBuilder("from" + upperCamel(typedValueName))
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addParameter(typedValueParameter)
                .returns(enumName)
                .beginControlFlow("if ($L == null)", typedValueName)
                .addStatement("return null")
                .endControlFlow()
                .addStatement("$T $L = $L.get($L)", enumName, resultValueName, enumMapName, typedValueName)
                .beginControlFlow("if ($L == null)", resultValueName)
                .addStatement("throw new $T($S + $L)", IllegalArgumentException.class,
                        "Unknown enum typed value ", typedValueName)
                .endControlFlow()
                .addStatement("return $L", resultValueName)
                .build();

        enumBuilder.addField(typedValuesType, typedValueName, Modifier.PRIVATE, Modifier.FINAL)
                .addField(enumMapField)
                .addMethod(constructor)
                .addMethod(typedValueGetter)
                .addMethod(fromTypedValueConverterMethod);
    }
}
