package ru.yandex.direct.model.generator.old.javafile;

import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.lang.model.element.Modifier;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.conf.AnnotationConf;
import ru.yandex.direct.model.generator.old.conf.AttrConf;
import ru.yandex.direct.model.generator.old.spec.InterfaceSpec;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.ALL_MODEL_PROPERTIES_METHOD_NAME;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.ATTR_TO_MODEL_PROPERTY_NAME;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.ID_FIELD;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.IMMUTABLE_SET;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.MODEL;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.MODEL_PROPERTY;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.MODEL_WITH_ID;
import static ru.yandex.direct.model.generator.old.javafile.ModelClassBuilder.RETURN_OF_ALL_MODEL_PROPERTIES_METHOD;
import static ru.yandex.direct.model.generator.old.javafile.Util.makeJsonSubtypes;

class ModelInterfaceBuilder {

    private final InterfaceSpec spec;

    private final Map<AttrConf, MethodSpec> getters = new HashMap<>();
    private final Map<AttrConf, MethodSpec> setters = new HashMap<>();

    ModelInterfaceBuilder(InterfaceSpec spec) {
        this.spec = spec;
    }

    @SuppressWarnings("CheckReturnValue")
    TypeSpec build() {
        TypeSpec.Builder interfaceBuilder = TypeSpec.interfaceBuilder(spec.getClassName())
                .addAnnotation(Util.createGeneratedAnnotation(spec))
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc(spec.getComment());

        for (AnnotationConf annotationConf : spec.getAnnotations()) {
            interfaceBuilder.addAnnotation(annotationConf.toSpec());
        }

        if (!spec.getJsonSubtypes().isEmpty()) {
            interfaceBuilder.addAnnotation(makeJsonSubtypes(spec.getJsonSubtypes(), spec.isJsonSubtypesWithNameValue()));
        }

        Set<TypeName> superInterfaces = new LinkedHashSet<>(spec.getSuperInterfaces());
        superInterfaces.add(hasInterfaceIdField() ? MODEL_WITH_ID : MODEL);
        interfaceBuilder.addSuperinterfaces(superInterfaces);

        if (spec.isGenerateProperties()) {
            spec.getAttrs()
                    .forEach(attr -> interfaceBuilder.addField(makeProperty(attr)));
        }

        for (AttrConf attrConf : spec.getAttrs()) {
            checkNotNull(attrConf, "No such attribute in Model class: %s", attrConf.getName());
            interfaceBuilder.addMethod(getterFor(attrConf));
            if (!spec.isReadOnly()) {
                interfaceBuilder
                        .addMethod(setterFor(attrConf))
                        .addMethod(makeWithSetter(attrConf, false));
            }
        }

        spec.getInheritedAttributes().forEach(attr ->
                interfaceBuilder.addMethod(makeWithSetter(attr, true)));

        if (spec.isGenerateGetPropsMethod()) {
            interfaceBuilder.addMethod(makeGetModelPropertiesMethod());
        }

        return interfaceBuilder.build();
    }

    private FieldSpec makeProperty(AttrConf attr) {
        final CodeBlock initializer;
        if (spec.isReadOnly()) {
            initializer = CodeBlock.of("\n $1T.createReadOnly($2N.class, $3S, $2N::$4N)",
                    MODEL_PROPERTY, spec.getName(),
                    attr.getName(), getterFor(attr));
        } else {
            initializer = CodeBlock.of("\n $1T.create($2N.class, $3S, $2N::$4N, $2N::$5N)",
                    MODEL_PROPERTY, spec.getName(),
                    attr.getName(), getterFor(attr), setterFor(attr));
        }
        return FieldSpec.builder(
                ParameterizedTypeName.get(MODEL_PROPERTY, spec.getClassName(), attr.getType()),
                ATTR_TO_MODEL_PROPERTY_NAME.apply(attr),
                Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
                .addJavadoc(attr.getComment() != null ? attr.getComment() : "")
                .initializer(initializer)
                .build();
    }

    private MethodSpec setterFor(AttrConf attr) {
        return setters.computeIfAbsent(attr, this::makeSetter);
    }

    private MethodSpec makeSetter(AttrConf attr) {
        return MethodSpec.methodBuilder("set" + Util.upperCamel(attr.getName()))
                .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
                .addParameter(attr.getType(), attr.getName())
                .build();
    }

    private MethodSpec getterFor(AttrConf attr) {
        return getters.computeIfAbsent(attr, this::makeGetter);
    }

    private MethodSpec makeGetter(AttrConf attr) {
        MethodSpec.Builder builder = MethodSpec.methodBuilder("get" + Util.upperCamel(attr.getName()))
                .addJavadoc(attr.getComment())
                .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
                .returns(attr.getType());

        attr.getAnnotations().stream()
                .filter(AnnotationConf::annotateGetter)
                .forEach(c -> builder.addAnnotation(c.toSpec()));

        return builder
                .build();
    }

    private MethodSpec makeWithSetter(AttrConf attr, boolean override) {
        MethodSpec.Builder builder = MethodSpec.methodBuilder("with" + Util.upperCamel(attr.getName()))
                .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
        if (override) {
            builder.addAnnotation(Override.class);
        }
        return builder
                .addParameter(attr.getType(), attr.getName())
                .returns(spec.getClassName())
                .build();
    }

    private MethodSpec makeGetModelPropertiesMethod() {
        List<String> fields = StreamEx.of(spec.getInheritedAttributes())
                .append(spec.getAttrs())
                .map(ATTR_TO_MODEL_PROPERTY_NAME)
                .toList();

        MethodSpec.Builder builder = MethodSpec.methodBuilder(ALL_MODEL_PROPERTIES_METHOD_NAME)
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(RETURN_OF_ALL_MODEL_PROPERTIES_METHOD)
                .addStatement("return $T.of(\n$L\n)",
                        IMMUTABLE_SET,
                        String.join(",\n", fields)
                );
        return builder.build();
    }

    /**
     * имеет ли интерфейс поле id типа Long среди собственных и отнаследованных аттрибутов
     */
    private boolean hasInterfaceIdField() {
        return StreamEx.of(spec.getInheritedAttributes())
                .append(spec.getAttrs().stream())
                .anyMatch(attr -> attr.getName().equals(ID_FIELD) && attr.getType().equals(TypeName.get(Long.class)));
    }
}
