package ru.yandex.direct.model.generator.old.javafile;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Generated;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.google.common.base.CharMatcher;
import com.google.common.base.Splitter;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.typesafe.config.Config;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.generator.Tool;
import ru.yandex.direct.model.generator.old.spec.JavaFileSpec;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static com.google.common.base.CaseFormat.UPPER_UNDERSCORE;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

@ParametersAreNonnullByDefault
public class Util {
    private static final Logger logger = LoggerFactory.getLogger(Util.class);

    private static final Splitter PARAMS_SPLITTER =
            Splitter.on(CharMatcher.whitespace().or(CharMatcher.is(','))).omitEmptyStrings();

    private static final Map<String, TypeName> PRIMITIVE_TYPES = Map.of(
            "byte", TypeName.BYTE,
            "char", TypeName.CHAR,
            "short", TypeName.SHORT,
            "int", TypeName.INT,
            "long", TypeName.LONG,

            "float", TypeName.FLOAT,
            "double", TypeName.DOUBLE,

            "boolean", TypeName.BOOLEAN
    );

    private static final String[] DEFAULT_PACKAGES =
            {"java.lang", "java.util", "java.time", "java.math", "org.jooq.types"};
    private static final ClassName[] DEFAULT_CLASSES = {ModelClassBuilder.MODEL, ModelClassBuilder.MODEL_WITH_ID};

    private static final Pattern GENERIC_TYPE_PATTERN = Pattern.compile("^(\\S+?)\\s*<\\s*(.*?)\\s*>$");
    private static final Pattern CLASS_NAME_PATTERN = Pattern.compile("^([^<>]*)\\.([^<>]*)$");
    private static final Pattern ANNOTATION_PATTERN = Pattern.compile("^@([a-zA-Z0-9_.]+)*?[a-zA-Z0-9_]+$");

    public static final Pattern IMPORTS_PATTERN =
            Pattern.compile("^(.*?)\n(import [^\n]+\n(?:import [^\n]+\n|\\s*\n)*)" + "(.*)$",
                    Pattern.DOTALL);

    private Util() {
    }

    static String upperUnderscore(String name) {
        return LOWER_CAMEL.to(UPPER_UNDERSCORE, name);
    }

    static String upperCamel(String name) {
        return LOWER_CAMEL.to(UPPER_CAMEL, name);
    }

    /**
     * Создать тип для javapoet по строке. Пытается поддерживать дженерики, но делает это плохо -
     * не понимает "?" и "," во вложенных дженериках (List&lt;Map&lt;Long, Long>>)
     */
    @Nonnull
    public static TypeName typeNameOf(String type, @Nullable String defaultPackage) {
        if (PRIMITIVE_TYPES.containsKey(type)) {
            return PRIMITIVE_TYPES.get(type);
        }
        if (type.endsWith("[]")) {
            return ArrayTypeName.of(typeNameOf(type.substring(0, type.length() - 2), defaultPackage));
        }
        Matcher matcher = GENERIC_TYPE_PATTERN.matcher(type);
        if (matcher.find()) {
            String mainType = matcher.group(1);

            List<TypeName> typeNameList = new ArrayList<>();
            List<String> buffer = new ArrayList<>();
            List<AnnotationSpec> annotations = new ArrayList<>();
            for (String param : PARAMS_SPLITTER.split(matcher.group(2))) {
                // Если у нас еще не встречались неаннотации, а текущий параметр - аннотация, добавляем его в список
                // и затем добавляем к типу
                if (buffer.isEmpty() && isAnnotation(param)) {
                    AnnotationSpec annotationSpec =
                            AnnotationSpec.builder(classNameOf(param.substring(1), defaultPackage)).build();
                    annotations.add(annotationSpec);
                    continue;
                }

                buffer.add(param);
                try {
                    TypeName typeName = typeNameOf(String.join(", ", buffer), defaultPackage)
                            .annotated(annotations);
                    typeNameList.add(typeName);

                    annotations.clear();
                    buffer.clear();
                } catch (IllegalArgumentException e) {
                    continue;
                }
            }

            return ParameterizedTypeName
                    .get(classNameOf(mainType, defaultPackage), typeNameList.toArray(new TypeName[0]));
        } else {
            return classNameOf(type, defaultPackage);
        }
    }

    private static boolean isAnnotation(String literal) {
        return ANNOTATION_PATTERN.matcher(literal).matches();
    }

    /**
     * Создание ClassName для javapoet (сам класс не обязан существовать).
     * Для full-qualified имени класса - просто его создаёт. Иначе пытается найти подходящий класс в пакетах
     * DEFAULT_PACKAGES и среди классов DEFAULT_CLASSES. Если не находит - использует defaultPackage.
     *
     * @throws IllegalArgumentException - если класс не fq, не нашёлся в списках и defaultPackage не определён
     */
    @Nonnull
    public static ClassName classNameOf(String className, @Nullable String defaultPackage) {
        Matcher matcher = CLASS_NAME_PATTERN.matcher(className);
        if (matcher.find()) {
            return ClassName.get(matcher.group(1), matcher.group(2));
        } else {
            for (String packageName : DEFAULT_PACKAGES) {
                if (classExists(packageName, className)) {
                    return ClassName.get(packageName, className);
                }
            }
            for (ClassName defaultClass : DEFAULT_CLASSES) {
                if (className.equals(defaultClass.simpleName())) {
                    return defaultClass;
                }
            }
            if (defaultPackage == null) {
                throw new IllegalArgumentException("Unknown class: " + className);
            } else {
                return ClassName.get(defaultPackage, className);
            }
        }
    }

    /**
     * Существует ли класс
     */
    private static boolean classExists(String packageName, String className) {
        try {
            Class.forName(packageName + "." + className);
            return true;
        } catch (ClassNotFoundException e) {
            logger.trace("No such class", e);
            return false;
        }
    }

    /**
     * Простенький хэлпер для работы со списками конфигов
     */
    @Nonnull
    public static <T> List<T> configObjects(Config config, String key, Function<Config, T> mapper) {
        if (!config.hasPath(key)) {
            return emptyList();
        }
        return config.getConfigList(key).stream()
                .map(mapper)
                .collect(toList());
    }

    public static AnnotationSpec makeJsonSubtypes(List<TypeName> jsonSubtypes, boolean useNameValueParameters) {
        CodeBlock.Builder codeBuilder = CodeBlock.builder();
        for (int i = 0; i < jsonSubtypes.size(); i++) {
            codeBuilder.add(i == 0 ? "{\n" : ",\n");
            if (useNameValueParameters)
                codeBuilder.add("@$T(value = $T.class, name = \"$T\")", JsonSubTypes.Type.class, jsonSubtypes.get(i), jsonSubtypes.get(i));
            else
                codeBuilder.add("@$T($T.class)", JsonSubTypes.Type.class, jsonSubtypes.get(i));
        }
        codeBuilder.add("\n}");
        return AnnotationSpec.builder(JsonSubTypes.class)
                .addMember("value", codeBuilder.build())
                .build();
    }

    public static AnnotationSpec createGeneratedAnnotation(JavaFileSpec spec) {
        AnnotationSpec.Builder builder = AnnotationSpec.builder(Generated.class)
                .addMember("value", "$S", Tool.class.getCanonicalName());
        if (spec.getSourceFileName() != null) {
            Path sourceFileName = Paths.get(spec.getSourceFileName()).getFileName();
            builder.addMember("comments", "$S", "generated from " + sourceFileName);
        }
        return builder.build();
    }

    /**
     * Подрезать хвостовые пробелы в строках (чтобы jstyle не ругался)
     */
    public static String trimTrailingSpaces(String javaCode) {
        return linesStream(javaCode)
                .map(String::stripTrailing)
                .joining("\n");
    }

    /**
     * Перереупорядочить импорты в тексте java-кода согласно yandex-codestyle
     */
    public static String rearrangeImports(String javaCode) {
        Matcher matcher = IMPORTS_PATTERN.matcher(javaCode);
        if (!matcher.find()) {
            return javaCode;
        }

        return matcher.group(1) + "\n"
                + rearrangeImportsInternal(matcher.group(2))
                + matcher.group(3);
    }

    // Правила взяты из секции ImportOrder отсюда
    // https://a.yandex-team.ru/arc/trunk/arcadia/devtools/jstyle-runner/java/resources/yandex_checks_extended.xml
    private static String rearrangeImportsInternal(String imports) {
        Map<String, String> grp = linesStream(imports)
                .filter(s -> !StringUtils.isBlank(s))
                .sorted()
                .groupingBy(line -> {
                    if (line.startsWith("import java.")) {
                        return "java";
                    } else if (line.startsWith("import javax.")) {
                        return "javax";
                    } else if (line.startsWith("import ru.yandex.") || line.startsWith("import yandex.")) {
                        return "yandex";
                    } else if (line.startsWith("import static ")) {
                        return "static";
                    } else {
                        return "other";
                    }
                }, Collectors.joining("\n", "", "\n"));

        StringBuilder res = new StringBuilder();
        Optional.ofNullable(grp.get("java"))
                .ifPresent(s -> res.append(s).append('\n'));
        Optional.ofNullable(grp.get("javax"))
                .ifPresent(s -> res.append(s).append('\n'));
        Optional.ofNullable(grp.get("other"))
                .ifPresent(s -> res.append(s).append('\n'));
        Optional.ofNullable(grp.get("yandex"))
                .ifPresent(s -> res.append(s).append('\n'));
        Optional.ofNullable(grp.get("static"))
                .ifPresent(s -> res.append(s).append('\n'));

        return res.toString();
    }

    private static StreamEx<String> linesStream(String source) {
        return StreamEx.of(Splitter.on('\n').split(source).iterator());
    }
}
