package ru.yandex.direct.model.generator.old.registry;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.conf.UpperLevelModelConf;

/**
 * Класс для обхода описаний {@link UpperLevelModelConf}
 * по иерархиям наследования вверх и вниз.
 * Родительскими описаниями считаются
 * <ul>
 * <li>описания суперкласса и всех реализуемых интерфейсов – для классов,</li>
 * <li>описания всех наследуемых интерфейсов – для интерфейсов.</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
class ClassTree {

    private final Multimap<UpperLevelModelConf, UpperLevelModelConf> parents = HashMultimap.create();
    private final Multimap<UpperLevelModelConf, UpperLevelModelConf> children = HashMultimap.create();
    private final Set<UpperLevelModelConf> leaves;

    ClassTree(Collection<UpperLevelModelConf> confCollection) {
        Map<String, UpperLevelModelConf> fullNameIndex =
                Maps.uniqueIndex(confCollection, UpperLevelModelConf::getFullName);

        StreamEx.of(confCollection)
                .mapToEntry(UpperLevelModelConf::getFullName, UpperLevelModelConf::getExtendsAndImplementsFullNames)
                .flatMapValues(Collection::stream)
                .mapKeys(fullNameIndex::get)
                .mapValues(fullNameIndex::get)
                .peekKeyValue(parents::put)
                .invert()
                .forKeyValue(children::put);

        Set<UpperLevelModelConf> leavesSet = new HashSet<>(confCollection);
        leavesSet.removeAll(children.keySet());
        this.leaves = Collections.unmodifiableSet(leavesSet);
    }

    /**
     * @return Все описания {@link UpperLevelModelConf}, у которых нет наследников.
     */
    Set<UpperLevelModelConf> getAllLeaves() {
        return new HashSet<>(leaves);
    }

    /**
     * Получить описания всех наследников класса/интерфейса.
     *
     * @param conf описание класса/интерфейса, {@link UpperLevelModelConf}
     * @return Для класса вернутся все субклассы; для интерфейса –
     * все интерфейсы-наследники и имплементации.
     */
    Set<UpperLevelModelConf> getAllDescendants(UpperLevelModelConf conf) {
        return getAllDescendants(conf, new LinkedHashSet<>());
    }

    private Set<UpperLevelModelConf> getAllDescendants(UpperLevelModelConf conf, Set<UpperLevelModelConf> foundSoFar) {
        for (UpperLevelModelConf child : children.get(conf)) {
            if (child != null && foundSoFar.add(child)) {
                getAllDescendants(child, foundSoFar);
            }
        }
        return foundSoFar;
    }

    /**
     * Получить описания всех предков класса/интерфейса.
     *
     * @param conf описание класса/интерфейса, {@link UpperLevelModelConf}
     * @return Для класса вернутся его суперклассы и реализуемые
     * интерфейсы, для интерфейса – его суперы.
     */
    Set<UpperLevelModelConf> getAllAncestors(UpperLevelModelConf conf) {
        return getAllAncestors(conf, new LinkedHashSet<>());
    }

    private Set<UpperLevelModelConf> getAllAncestors(UpperLevelModelConf conf, Set<UpperLevelModelConf> foundSoFar) {
        for (UpperLevelModelConf parent : parents.get(conf)) {
            if (parent != null && foundSoFar.add(parent)) {
                getAllAncestors(parent, foundSoFar);
            }
        }
        return foundSoFar;
    }

    /**
     * Из заданного множества описаний выделить старшие, т.е. те, у кого нет родителей среди заданных.
     *
     * @param given заданное множество описаний классов/интерфейсов
     * @return Множество, состоящее только из элементов данного, обладающее свойствами:
     * <ol>
     * <li>не содержит пар элементов,связанных наследованием.</li>
     * <li>отфильтровано редуцированием к описаниям старших классов/интерфейсов</li>
     * </ol>
     */
    Set<UpperLevelModelConf> filterUpperClassesAmong(Collection<UpperLevelModelConf> given) {
        Set<UpperLevelModelConf> givenSet = new HashSet<>(given);
        return StreamEx.of(given)
                .filter(conf -> parents.get(conf).stream().noneMatch(givenSet::contains))
                .toSet();
    }

}
