package ru.yandex.direct.model.generator.old.registry;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.conf.InterfaceConf;
import ru.yandex.direct.model.generator.old.conf.ModelClassConf;
import ru.yandex.direct.model.generator.old.conf.ModelConf;
import ru.yandex.direct.model.generator.old.conf.UpperLevelModelConf;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;

/**
 * Хранилище (таблица) обрабатываемых {@link UpperLevelModelConf} объектов,
 * то есть описаний модели верхнего уровня.
 * <p>
 * Предоставляет информацию об иерархии наследования моделей.
 *
 * @see #getPropertyHolder
 * @see #getAllAncestors(ModelConf)
 */
@ParametersAreNonnullByDefault
public class ModelConfRegistry {

    private final Map<String, ModelConf> fullNameIndex;

    private final Map<String, Set<ModelConf>> subtypes;

    private final ModelPropertiesResolver propertiesResolver;

    public ModelConfRegistry(Collection<UpperLevelModelConf> modelConfigs) {
        fullNameIndex =
                Maps.uniqueIndex(flatMap(modelConfigs, UpperLevelModelConf::getAllModelConfs), ModelConf::getFullName);
        propertiesResolver = new ModelPropertiesResolver(modelConfigs);

        subtypes = new HashMap<>();
        for (UpperLevelModelConf conf : modelConfigs) {
            for (String superClass : conf.getExtendsAndImplementsFullNames()) {
                //Используется LinkedHashSet, т.к. в тестах используется line-by-line сравнение,
                //поэтому нужно гарантировать порядок
                subtypes.computeIfAbsent(superClass, s -> new LinkedHashSet<>()).add(conf);
            }
        }
    }

    /**
     * <p>
     * Получить описание вспомогательного property holder интерфейса {@link PropertyHolderMeta},
     * соответствующее данному атрибуту.
     * <p>
     * При нескольких вызовах метода с разными именами атрибутов в случае, если они представляют
     * одно свойство модели ({@link ru.yandex.direct.model.ModelProperty}), будет возвращаться
     * тот же инстанс {@link PropertyHolderMeta}.
     *
     * @param attrFullName полное имя атрибута
     * @return Описание соответствующего property holder интерфейса.
     */
    public PropertyHolderMeta getPropertyHolder(String attrFullName) {
        return propertiesResolver.getPropertyHolder(attrFullName);
    }

    /**
     * Получить все конфиги.
     *
     * @return Все занесённые в хранилище конфиги.
     */
    public Set<ModelConf> getAllModelConfigs() {
        return new HashSet<>(fullNameIndex.values());
    }

    /**
     * @return Все описания property holder интерфейсов.
     */
    public Set<PropertyHolderMeta> getAllPropertyHoldersMeta() {
        return propertiesResolver.getAllPropertyHolders();
    }

    public Set<ModelConf> getSubtypes(ModelConf conf) {
        return subtypes.getOrDefault(conf.getFullName(), Collections.emptySet());
    }

    /**
     * Получить описания всех предков модели.
     *
     * @param conf описание некоторой модели.
     * @return Описания всех моделей, субклассом которых является заданная.
     * Если передан {@link ModelClassConf} модели, не занесённой ранее в хранилище,
     * вернётся пустой {@link Set}.
     * <p>
     * <i>Для классов вернутся только суперклассы, без реализуемых интерфейсов.</i>
     */
    public Set<ModelConf> getAllAncestors(ModelConf conf) {
        return getAllAncestors(conf, new LinkedHashSet<>());
    }

    /**
     * Получить всех предков класса
     *
     * @param conf конфиг класса
     */
    public Set<ModelClassConf> getClassAncestors(ModelClassConf conf) {
        //Используется LinkedHashSet, т.к. в тестах используется line-by-line сравнение,
        //поэтому нужно гарантировать порядок
        return StreamEx.of(getAllAncestors(conf))
                .peek(c -> checkState(c instanceof ModelClassConf, "ancestor of %s must be ModelClassConf, but for %s was %s",
                        conf.getName(), c.getName(), c.getClass()))
                .map(c -> (ModelClassConf) c)
                .collect(Collectors.toCollection(LinkedHashSet::new));
    }

    /**
     * Получить конфиг класса {@link ModelClassConf}, в котором был объявлен интерфейс {@link InterfaceConf}
     *
     * @param interfaceConf конфиг интерфейса
     */
    public ModelClassConf getInterfaceModelClassConf(InterfaceConf interfaceConf) {
        ModelConf modelConf = getSubtypes(interfaceConf)
                .stream()
                .filter(t -> t.getFullName().equals(interfaceConf.getClassConfigFullName()))
                .findFirst()
                .orElse(null);
        checkNotNull(modelConf, "not found class %s for interface %s", interfaceConf.getClassConfigFullName(),
                interfaceConf.getName());
        checkState(modelConf instanceof ModelClassConf, "class %s not instance of ModelClassConf", modelConf.getName());
        return (ModelClassConf) modelConf;
    }

    /**
     * Имеет ли предок класса свойство {@link ModelClassConf#isGenerateCopyMethod()} == true
     */
    public boolean isClassAncestorsHasCopyMethod(ModelClassConf conf) {
        return getAllAncestors(conf).stream()
                .map(ModelClassConf.class::cast)
                .anyMatch(ModelClassConf::isGenerateCopyMethod);
    }

    private Set<ModelConf> getAllAncestors(ModelConf conf, Set<ModelConf> ancestors) {
        for (String superName : conf.getSupersFullNames()) {
            if (fullNameIndex.containsKey(superName)) {
                ModelConf superConf = fullNameIndex.get(superName);
                if (ancestors.add(superConf)) {     // else: already visited, terminate
                    getAllAncestors(superConf, ancestors);
                }
            }
        }
        return ancestors;   // termination
    }

}
