package ru.yandex.direct.model.generator.old.registry;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.TreeMultimap;
import com.squareup.javapoet.TypeName;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.conf.AttrConf;
import ru.yandex.direct.model.generator.old.conf.ModelConf;
import ru.yandex.direct.model.generator.old.conf.UpperLevelModelConf;
import ru.yandex.direct.model.generator.old.util.AttrNameUtil;
import ru.yandex.direct.model.generator.uf.UnionFind;
import ru.yandex.direct.model.generator.uf.UnionFindFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Comparator.comparing;
import static java.util.function.Function.identity;
import static ru.yandex.direct.model.generator.old.util.AttrNameUtil.getAttrFullName;

/**
 * <p>
 * Класс для обхода всех моделей и выделения среди них множеств
 * {@link ru.yandex.direct.model.ModelProperty}, равных между собой.
 * <p>
 * После успешного создания {@link #getPropertyHolder} будет возвращать объект,
 * описывающий property holder интерфейс {@link PropertyHolderMeta},
 * соответствующий атрибуту.
 */
@ParametersAreNonnullByDefault
class ModelPropertiesResolver {

    private final PropertyHolderMetaFactory propertyHoldersFactory = new PropertyHolderMetaFactory();

    /**
     * Полное имя атрибута -> короткое имя атрибута. Чтобы не вычислять по нескольку раз.
     */
    private final Map<String, String> fullToShortName = new HashMap<>();

    /**
     * Мультимапа: описание модели –> полные имена явно объявленных в нём атрибутов.
     * <p>
     * Хранит записи в отсортированном виде во имя детерминизма:
     * чтобы от сборки к сборке сохранялся предсказуемый порядок обработки,
     * в конечном счёте это влияет на названия property holder-ов.
     */
    private final Multimap<UpperLevelModelConf, String> declaratorAttrs = TreeMultimap.create(
            comparing(UpperLevelModelConf::getFullName), String::compareTo);

    /**
     * Полное имя атрибута -> название типа атрибута.
     * Здесь не все атрибуты, а только те, для которых был {@link AttrConf}.
     */
    private final Map<String, TypeName> attrFullNameToTypeName = new HashMap<>();

    /**
     * Идентификатор множества эквивалентности атрибутов (идентификатор одного свойства модели) –>
     * тип атрибутов этого множества.
     */
    private final Map<Integer, TypeName> propIdToTypeName = new HashMap<>();

    /**
     * Идентификатор свойства модели –> соответствующее описание property holder интерфейса.
     */
    private final Map<Integer, PropertyHolderMeta> propertyHolders = new HashMap<>();

    private final ClassTree classTree;
    private final UnionFind<String> attrUF;

    ModelPropertiesResolver(Collection<UpperLevelModelConf> modelConfigs) {
        classTree = new ClassTree(modelConfigs);

        modelConfigs.forEach(this::include);

        attrUF = UnionFindFactory.createUnionFind(declaratorAttrs.values());

        resolve();
        createPropertyHolders();
        postValidateIntegralConsistency();
    }

    PropertyHolderMeta getPropertyHolder(String attributeFullName) {
        int propertyId = getPropertyId(attributeFullName);
        return propertyHolders.get(propertyId);
    }

    Set<PropertyHolderMeta> getAllPropertyHolders() {
        return new HashSet<>(propertyHolders.values());
    }

    /**
     * <p>
     * Для каждого атрибута модели, описываемой объектом {@link UpperLevelModelConf}:
     * <ul>
     * <li>Занести полное и короткое имена в мапу {@link #fullToShortName};</li>
     * <li>Занести в мапу {@link #declaratorAttrs} его декларатора
     * (описание, где он объявлен), и его полное имя;</li>
     * <li>Поскольку во всех наследниках этот атрибут тоже присутствует,
     * также деклараторов-наследников и полные имена унаследованных атрибутов
     * в мапу {@link #declaratorAttrs}.</li>
     * </ul>
     * <p>
     * Мапа {@link #fullToShortName} используется, чтобы не вычислять короткое имя по нескольку раз.
     * <p>
     * В мапе {@link #declaratorAttrs} оказываются абсолютно все атрибуты,
     * включая неявно унаследованные, по каждой модели {@link UpperLevelModelConf}.
     */
    @SuppressWarnings("CheckReturnValue")
    private void include(UpperLevelModelConf conf) {
        Multimap<String, String> nestedModelsAttrNames = Multimaps.index(
                conf.getNestedModelsAttributesFullPaths(), AttrNameUtil::getAttrShortName);

        for (AttrConf attrConf : conf.getAttrs()) {
            String attrFullName = getAttrFullName(attrConf, conf);
            String attrShortName = attrConf.getName();

            fullToShortName.put(attrFullName, attrShortName);
            attrFullNameToTypeName.put(attrFullName, attrConf.getType());

            declaratorAttrs.put(conf, attrFullName);
            StreamEx.of(classTree.getAllDescendants(conf))
                    .mapToEntry(identity(), descendant -> AttrNameUtil.getAttrFullName(attrConf, descendant))
                    .forKeyValue(declaratorAttrs::put);

            // запоминаем атрибуты вложенных интерфейсов
            if (nestedModelsAttrNames.containsKey(attrShortName)) {
                for (String nestedModelAttrFullName : nestedModelsAttrNames.get(attrShortName)) {
                    fullToShortName.put(nestedModelAttrFullName, attrShortName);
                    attrFullNameToTypeName.put(nestedModelAttrFullName, attrConf.getType());
                    // записываем их на этот же conf
                    declaratorAttrs.put(conf, nestedModelAttrFullName);
                }
            }
        }
    }

    /**
     * От каждого листа иерархии идём вверх, объединяя атрибуты в непересекающиеся множества.
     */
    private void resolve() {
        StreamEx.of(classTree.getAllLeaves())
                .mapToEntry(identity(), declaratorAttrs::get)
                .flatMapValues(Collection::stream)
                .forKeyValue(this::processAttributeOfParticularDeclarator);
    }

    /**
     * <p>
     * Берём полное имя атрибута, идём вверх по иерархии от её декларатора,
     * находим атрибуты, короткие имена которых совпадают с коротким именем
     * исходного атрибута, объединяем их в одно множество – они представляют
     * одно свойство модели.
     * <p>
     * Кроме этого, запоминается тип проперти, {@link #propIdToTypeName}.
     * <p>
     * В этот метод достаточно передавать только модели, являющиеся листьями
     * иерархии наследования, поскольку здесь обрабатываются все классы
     * и интерфейсы вверх по иерархии.
     *
     * @throws IllegalStateException если не удалось определить тип проперти.
     */
    private void processAttributeOfParticularDeclarator(UpperLevelModelConf declarator, String attrFullName) {
        String attrShortName = fullToShortName.computeIfAbsent(attrFullName, AttrNameUtil::getAttrShortName);

        for (UpperLevelModelConf ancestor : classTree.getAllAncestors(declarator)) {
            for (String superAttrFullName : ancestor.getAttributesFullPaths()) {
                // атрибуты представляют одну пропертю, если их короткие имена совпадают
                // в пределах одной иерархии классов
                if (attrShortName.equals(fullToShortName.get(superAttrFullName))) {
                    attrUF.connect(attrFullName, superAttrFullName);    // объединяем множества

                    // определяем и запоминаем тип проперти
                    TypeName attrType = attrFullNameToTypeName.get(superAttrFullName);
                    if (attrType != null) {
                        propIdToTypeName.putIfAbsent(getPropertyId(superAttrFullName), attrType);
                    }
                }
            }
        }

        propIdToTypeName.putIfAbsent(getPropertyId(attrFullName), attrFullNameToTypeName.get(attrFullName));

        checkState(propIdToTypeName.containsKey(getPropertyId(attrFullName)),
                "Could not determine type of attribute: %s", attrFullName);
    }

    /**
     * Создать PropertyHolder-ов после того, как все атрибуты разбиты на множества.
     * По одному на множество, то есть на каждую проперти.
     */
    @SuppressWarnings("CheckReturnValue")
    private void createPropertyHolders() {
        Map<Integer, String> propIdToShortName = new HashMap<>();
        Multimap<Integer, UpperLevelModelConf> propIdToDeclarators = HashMultimap.create();

        for (UpperLevelModelConf conf : declaratorAttrs.keySet()) {
            if (!isImplementsOnlyPropHolders(conf) || conf.getType() != ModelConf.Type.CLASS) {
                StreamEx.of(conf.getAttributesFullPaths())
                        .mapToEntry(this::getPropertyId)
                        .peekKeyValue((attr, id) -> propIdToShortName.put(id, fullToShortName.get(attr)))
                        .forKeyValue((attr, id) -> propIdToDeclarators.put(id, conf));
            }
        }

        for (Map.Entry<Integer, String> e : propIdToShortName.entrySet()) {
            Integer propertyId = e.getKey();
            String attrShortName = e.getValue();

            Set<UpperLevelModelConf> namingCandidates = classTree.filterUpperClassesAmong(
                    propIdToDeclarators.get(propertyId));

            UpperLevelModelConf bestNamingCandidate = StreamEx.of(namingCandidates)
                    .min(this::compareNames)
                    .orElseThrow(IllegalStateException::new);
            PropertyHolderMeta propertyHolder = propertyHoldersFactory.createPropertyHolder(
                    bestNamingCandidate.getClassName(),
                    attrShortName,
                    getType(propertyId),
                    bestNamingCandidate.isJavaFileBuilt());
            propertyHolders.putIfAbsent(propertyId, propertyHolder);
        }
    }

    /**
     * Проверка на ситуацию, когда все имплеменитрованные интерфейсы - PropHolder'ы
     */
    private boolean isImplementsOnlyPropHolders(UpperLevelModelConf conf) {
        for (String name : conf.getExtendsAndImplementsFullNames()) {
            if (!conf.getSupersFullNames().contains(name)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Сравнить два имени класса в качестве кандидатов на именование property holder интерфейса для некоторой
     * их общей проперти. Получается приоритет, т.е. меньше результат – лучше кандидат.
     * <p>
     * В первую очередь предпочитаются уже существующие классы
     * Лучшим считается класс/интерфейс с коротким названием.
     * Если длины равны, побеждает лексикографически старшее название.
     */
    private int compareNames(UpperLevelModelConf first, UpperLevelModelConf second) {
        if (first.isJavaFileBuilt() != second.isJavaFileBuilt()) {
            // если файл создан - предпочитаем его, отдаем меньший приоритет
            return first.isJavaFileBuilt() ? -1 : 1;
        }

        String firstName = first.getClassName().simpleName();
        String secondName = second.getClassName().simpleName();
        int lengthComp = Integer.compare(firstName.length(), secondName.length());
        return lengthComp == 0 ? firstName.compareTo(secondName) : lengthComp;
    }

    /**
     * Проверяем, нет ли коллизий по именам среди property holder-ов для разных типов пропертей.
     *
     * @throws IllegalStateException если обнаружены коллизии.
     */
    private void postValidateIntegralConsistency() {
        Multimap<String, PropertyHolderMeta> m =
                Multimaps.index(propertyHolders.values(), PropertyHolderMeta::getFullName);
        for (Collection<PropertyHolderMeta> holders : m.asMap().values()) {
            long distinctTypes = holders.stream()
                    .map(PropertyHolderMeta::getAttr)
                    .map(AttrConf::getType)
                    .distinct().count();
            checkState(distinctTypes == 1,
                    "Holders for props of different types clash by name: %s",
                    holders.iterator().next().getFullName());
        }
    }

    private int getPropertyId(String attributeFullName) {
        return attrUF.find(attributeFullName);
    }

    private TypeName getType(int propertyId) {
        return propIdToTypeName.get(propertyId).box();
    }

}
