package ru.yandex.direct.model.generator.old.spec.factory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.conf.AttrConf;
import ru.yandex.direct.model.generator.old.conf.InterfaceConf;
import ru.yandex.direct.model.generator.old.conf.ModelClassConf;
import ru.yandex.direct.model.generator.old.conf.ModelConf;
import ru.yandex.direct.model.generator.old.conf.ModelInterfaceConf;
import ru.yandex.direct.model.generator.old.javafile.Util;
import ru.yandex.direct.model.generator.old.registry.ModelConfRegistry;
import ru.yandex.direct.model.generator.old.registry.PropertyHolderMeta;
import ru.yandex.direct.model.generator.old.spec.InterfaceSpec;
import ru.yandex.direct.model.generator.old.spec.PropertyHolderInterfaceSpec;
import ru.yandex.direct.model.generator.old.util.AttrNameUtil;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
class InterfaceSpecFactory {

    private final ModelConfRegistry registry;

    InterfaceSpecFactory(ModelConfRegistry registry) {
        this.registry = registry;
    }

    InterfaceSpec createInterfaceSpec(ModelInterfaceConf conf) {

        List<TypeName> superInterfaces = new ArrayList<>();

        conf.getExtendsList().stream()
                .map(name -> Util.typeNameOf(name, conf.getPackageName()))
                .forEach(superInterfaces::add);

        List<AttrConf> inheritedAttrs = getInheritedAttrs(conf);

        if (conf.isGenerateProperties()) {  // extend property holders
            conf.getAttributesFullPaths().stream()
                    .map(registry::getPropertyHolder)
                    .filter(Objects::nonNull)
                    .map(PropertyHolderMeta::getClassName)
                    .forEach(superInterfaces::add);
        }

        List<TypeName> jsonSubtypes = conf.isJsonSubtypes() || conf.isJsonSubtypesWithNameValue()
                ? mapList(registry.getSubtypes(conf), ModelConf::getClassName)
                : emptyList();

        boolean generateGetPropsMethod = conf.isGenerateProperties();

        return new InterfaceSpec(conf, inheritedAttrs, superInterfaces, conf.getAnnotations(), jsonSubtypes,
                conf.isJsonSubtypesWithNameValue(), conf.isReadonly(), generateGetPropsMethod);
    }

    InterfaceSpec createInterfaceSpec(PropertyHolderMeta propertyHolderMeta) {
        return new PropertyHolderInterfaceSpec(propertyHolderMeta.getClassName(), propertyHolderMeta.getAttr());
    }

    InterfaceSpec createInterfaceSpec(InterfaceConf interfaceConf) {
        String packageName = interfaceConf.getPackageName();
        ClassName interfaceClassName = ClassName.get(packageName, interfaceConf.getName());
        ModelClassConf classConf = registry.getInterfaceModelClassConf(interfaceConf);
        List<TypeName> superInterfaces = new ArrayList<>();

        interfaceConf.getExtendsList().stream()
                .map(name -> Util.classNameOf(name, packageName))
                .forEach(superInterfaces::add);

        if (classConf.isGenerateProperties()) { // extend property holders
            interfaceConf.getAttrNames().stream()
                    .map(attr -> AttrNameUtil.getAttrFullName(attr, classConf))
                    .map(registry::getPropertyHolder)
                    .filter(Objects::nonNull)
                    .map(PropertyHolderMeta::getClassName)
                    .forEach(superInterfaces::add);
        }

        Map<String, AttrConf> classAttrs = getAllClassAttrs(classConf);
        List<AttrConf> attrs =
                mapList(interfaceConf.getAttrNames(), name -> getAttrWithCheck(classAttrs, name, interfaceConf));

        List<AttrConf> inheritedAttrs = getInheritedAttrs(interfaceConf);

        List<TypeName> jsonSubtypes = interfaceConf.isJsonSubtypes() || interfaceConf.isJsonSubtypesWithNameValue()
                ? Collections.singletonList(classConf.getClassName())
                : emptyList();

        boolean generateGetPropsMethod = classConf.isGenerateProperties();

        return new InterfaceSpec(interfaceConf, interfaceClassName, attrs, inheritedAttrs, superInterfaces,
                interfaceConf.getAnnotations(), jsonSubtypes, interfaceConf.isJsonSubtypesWithNameValue(),
                interfaceConf.isReadOnly(), generateGetPropsMethod);
    }

    /**
     * Находит все возможные атрибуты как в классе, так и среди всех предков
     *
     * @param classConf конфиг класса, в котором находится интерфейс
     */
    private Map<String, AttrConf> getAllClassAttrs(ModelClassConf classConf) {
        return StreamEx.of(registry.getClassAncestors(classConf))
                .flatCollection(ModelClassConf::getAttrs)
                .append(classConf.getAttrs())
                .distinct(AttrConf::getName)
                .toMap(AttrConf::getName, identity());
    }

    /**
     * Ищет все {@link AttrConf} связанные с атрибутами интерфейса
     * Сначала находит всех предков у интерфейса.
     * Потом все использования этих интерфейсов в классах.
     * Исключает атрибуты описанные в самом интерфейсе
     *
     * @param modelConf конфиг класса, в котором находится интерфейс. может быть {@link ModelInterfaceConf} или
     *                  {@link InterfaceConf}
     */
    private List<AttrConf> getInheritedAttrs(ModelConf modelConf) {
        Set<String> attrNames = new HashSet<>(modelConf.getAttrNames());
        return StreamEx.of(registry.getAllAncestors(modelConf))
                .flatCollection(this::getInterfaceAncestorAttrs)
                .distinct(AttrConf::getName)
                .remove(a -> attrNames.contains(a.getName()))
                .toList();
    }

    private List<AttrConf> getInterfaceAncestorAttrs(ModelConf conf) {
        List<AttrConf> attrs;
        if (conf instanceof ModelInterfaceConf) {
            // интерфейс сам себя описывает
            attrs = ((ModelInterfaceConf) conf).getAttrs();
        } else if (conf instanceof InterfaceConf) {
            // аттрибуты нужно взять из описания класса по которому формируется ограничивающий интерфейс
            attrs = registry.getInterfaceModelClassConf((InterfaceConf) conf).getAttrs();
        } else {
            throw new IllegalStateException("unexpected type of ModelConf: " + conf);
        }

        Map<String, AttrConf> upperLevelAttrs = StreamEx.of(attrs)
                .mapToEntry(AttrConf::getName)
                .invert()
                .toMap();

        return StreamEx.of(conf.getAttrNames())
                .map(name -> getAttrWithCheck(upperLevelAttrs, name, conf))
                .toList();
    }

    private AttrConf getAttrWithCheck(Map<String, AttrConf> classAttrs, String attrName, ModelConf conf) {
        return checkNotNull(classAttrs.get(attrName), "not found attr for name %s, config name: ", attrName,
                conf.getName());
    }
}
