package ru.yandex.direct.model.generator.rewrite

/**
 * Represents [Disjoin-set union](https://cp-algorithms.com/data_structures/disjoint_set_union.html) data structure
 */
interface DisjointSetUnion<E> {
    /**
     * Number of elements
     */
    fun size(): Int

    /**
     * Adds an element to the union
     *
     * @return `true` if the element was added
     */
    fun add(element: E): Boolean

    /**
     * Returns an index of the set, which [element] belongs to, or `null`
     */
    fun find(element: E): Int?

    /**
     * Unites sets holding elements [left] and [right]. Has no effect if [left] and [right] are already in the same set
     *
     * @return `false`, if [left] and [right] are already in the same set
     * @throws IllegalArgumentException if [left] or [right] do not belong in the union
     */
    fun merge(left: E, right: E): Boolean
}

class DisjointSetUnionImpl<E> : DisjointSetUnion<E> {
    private val indexes: MutableMap<E, Int> = mutableMapOf()
    private val parents: MutableList<Int> = mutableListOf()
    private val sizes: MutableList<Int> = mutableListOf()

    override fun size(): Int = indexes.size

    override fun add(element: E): Boolean {
        val newIndex = indexes.size
        val existingIndex = indexes.putIfAbsent(element, newIndex)
        if (existingIndex != null) {
            return false
        }
        parents.add(newIndex)
        sizes.add(1)
        return true
    }

    override fun find(element: E): Int? {
        val index = indexes[element]
            ?: return null
        return find(index)
    }

    private fun find(index: Int): Int {
        if (parents[index] == index) {
            return index
        }
        return find(parents[index])
            .also { result -> parents[index] = result }
    }

    override fun merge(left: E, right: E): Boolean {
        val leftIndex = find(left)
            ?: throw IllegalArgumentException("Element $left does not belong to set union")
        val rightIndex = find(right)
            ?: throw IllegalArgumentException("Element $left does not belong to set union")

        if (leftIndex == rightIndex) {
            return false
        }

        // attach smaller set to the larger one
        if (sizes[leftIndex] < sizes[rightIndex]) {
            parents[leftIndex] = parents[rightIndex]
            sizes[rightIndex] += sizes[leftIndex]
        } else {
            parents[rightIndex] = parents[leftIndex]
            sizes[leftIndex] += sizes[rightIndex]
        }

        return true
    }
}
