package ru.yandex.direct.model.generator.rewrite

import com.google.common.base.CharMatcher
import com.google.common.base.Splitter
import com.squareup.javapoet.AnnotationSpec
import com.squareup.javapoet.ArrayTypeName
import com.squareup.javapoet.ClassName
import com.squareup.javapoet.ParameterizedTypeName
import com.squareup.javapoet.TypeName
import ru.yandex.direct.model.Model
import ru.yandex.direct.model.ModelWithId

object JavaPoetUtils {

    private val paramsSplitter: Splitter =
        Splitter.on(CharMatcher.whitespace().or(CharMatcher.`is`(','))).omitEmptyStrings()

    private val classNamePattern: Regex = "^([^<>]*)\\.([^<>]*)$".toRegex()
    private val genericTypePattern: Regex = "^(\\S+?)\\s*<\\s*(.*?)\\s*>$".toRegex()
    private val annotationPattern: Regex = "^@([a-zA-Z0-9_.]+)*?[a-zA-Z0-9_]+$".toRegex()

    private val defaultPackages: Set<String> = setOf(
        "java.lang",
        "java.util",
        "java.time",
        "java.math",
        "org.jooq.types",
    )

    private val defaultClasses: Set<ClassName> = setOf(
        ClassName.get(Model::class.java),
        ClassName.get(ModelWithId::class.java),
    )

    private val primitiveTypes: Map<String, TypeName> = mapOf(
        "byte" to TypeName.BYTE,
        "char" to TypeName.CHAR,
        "short" to TypeName.SHORT,
        "int" to TypeName.INT,
        "long" to TypeName.LONG,
        "float" to TypeName.FLOAT,
        "double" to TypeName.DOUBLE,
        "boolean" to TypeName.BOOLEAN,
    )

    /**
     * Creates a [ClassName] from [className] string. If [className] is not qualified, it is searched for, in order:
     * - In packages [defaultPackage]
     * - In elements of [defaultClasses]
     *
     * Otherwise, if [defaultPackage] is provided, returned [ClassName] will have package [defaultPackage]
     *
     * @throws IllegalArgumentException if class is not found, and [defaultPackage] is `null`
     */
    fun toClassName(className: String, defaultPackage: String? = null): ClassName {
        val match = classNamePattern.matchEntire(className)
        if (match != null) {
            val (_, packageName, name) = match.groupValues
            return ClassName.get(packageName, name)
        }

        for (packageName in defaultPackages) {
            if (classExists(packageName, className)) {
                return ClassName.get(packageName, className)
            }
        }

        for (defaultClass in defaultClasses) {
            if (className == defaultClass.simpleName()) {
                return defaultClass
            }
        }

        if (defaultPackage != null) {
            return ClassName.get(defaultPackage, className)
        }

        throw IllegalArgumentException("Unknown class: $className")
    }

    private fun classExists(packageName: String, name: String): Boolean {
        return try {
            Class.forName("$packageName.$name")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Creates a [TypeName] from [typeName] string. Supports:
     * - Primitives
     * - Arrays of supported types
     * - Parameterized types (with annotated parameter types)
     * - Simple class names (as in [toClassName])
     *
     * @param defaultPackage will be used if result type is not qualified
     */
    fun toTypeName(typeName: String, defaultPackage: String? = null): TypeName {
        val primitive = primitiveTypes[typeName]
        if (primitive != null) {
            return primitive
        }

        if (typeName.endsWith("[]")) {
            val componentType = typeName.removeSuffix("[]")
            return ArrayTypeName.of(toTypeName(componentType, defaultPackage))
        }

        val match = genericTypePattern.matchEntire(typeName)
        if (match != null) {
            val (_, className, params) = match.groupValues
            val classType = toClassName(className, defaultPackage)
            val typeArguments = parseTypeArguments(params, defaultPackage)
            return ParameterizedTypeName.get(classType, *typeArguments.toTypedArray())
        }

        return toClassName(typeName, defaultPackage)
    }

    private fun parseTypeArguments(params: String, defaultPackage: String? = null): List<TypeName> {
        val typeArguments: MutableList<TypeName> = mutableListOf()

        val buffer: MutableList<String> = mutableListOf()
        val annotations: MutableList<AnnotationSpec> = mutableListOf()
        for (item in paramsSplitter.split(params)) {
            if (buffer.isEmpty() && annotationPattern.matches(item)) {
                val annotationName = item.removePrefix("@")
                val annotationClass = toClassName(annotationName, defaultPackage)
                annotations += AnnotationSpec.builder(annotationClass).build()
                continue
            }

            buffer += item

            val paramType = try {
                toTypeName(buffer.joinToString(", "), defaultPackage)
            } catch (e: IllegalArgumentException) {
                continue
            }

            typeArguments += paramType
                .annotated(annotations)

            annotations.clear()
            buffer.clear()
        }

        return typeArguments
    }

    fun toFullName(name: String, defaultPackage: String): String {
        return if (name.contains(".")) name else "$defaultPackage.$name"
    }
}
