package ru.yandex.direct.model.generator.rewrite

import com.google.common.collect.LinkedListMultimap
import com.google.common.collect.Multimap
import ru.yandex.direct.model.generator.rewrite.conf.Conf
import ru.yandex.direct.model.generator.rewrite.conf.UpperLevelConf

class ModelRegistry(confs: Collection<UpperLevelConf>) {
    private val confByName: Map<String, Conf>
    private val parents: Multimap<Conf, Conf> = LinkedListMultimap.create()
    private val children: Multimap<Conf, Conf> = LinkedListMultimap.create()

    init {
        confByName = confs
            .flatMap { conf -> conf.declaredConfs }
            .associateBy { it.fullName }

        confs.flatMap { it.declaredConfs }.forEach { conf ->
            conf.parents.forEach { parent ->
                confByName[parent]?.let { parentConf ->
                    parents.put(conf, parentConf)
                    children.put(parentConf, conf)
                }
            }
        }
    }

    fun allConfs(): Collection<Conf> =
        confByName.values

    fun getConf(className: String): Conf? =
        confByName[className]

    fun getSubtypes(conf: Conf): Collection<Conf> =
        children.get(conf).orEmpty()

    fun getAllDescendants(conf: Conf): Set<Conf> {
        return mutableSetOf<Conf>()
            .also { result -> collect(conf, children, result) }
    }

    fun getAllAncestors(conf: Conf): Set<Conf> {
        return mutableSetOf<Conf>()
            .also { result -> collect(conf, parents, result) }
    }

    /**
     * Recursively traverses all elements reachable by [edges] from [current] if depth-first order,
     * storing them in [result].
     */
    private fun <T : Conf> collect(current: T, edges: Multimap<T, T>, result: MutableCollection<T>) {
        if (result.add(current)) {
            edges[current].forEach { other ->
                collect(other, edges, result)
            }
        }
    }

    /**
     * Returns roots of DAG formed by [confs]. Subgraph formed by elements of [confs] has to be connected
     */
    fun findRoots(confs: Collection<Conf>): Set<Conf> {
        val confSet = confs.toSet()
        return confs
            .filter { conf ->
                parents[conf]
                    .none { parent -> parent in confSet }
            }
            .toSet()
    }
}

