package ru.yandex.direct.model.generator.rewrite

import com.google.common.collect.HashMultimap
import com.google.common.collect.Multimap
import com.squareup.javapoet.ClassName
import ru.yandex.direct.model.generator.rewrite.conf.AttributeConf
import ru.yandex.direct.model.generator.rewrite.conf.Conf
import ru.yandex.direct.model.generator.rewrite.conf.attributes
import ru.yandex.direct.model.generator.rewrite.conf.builtModuleName
import ru.yandex.direct.model.generator.rewrite.conf.isAlreadyBuilt

class PropertyResolver(private val registry: ModelRegistry) {

    /**
     * Comparator for choosing the best class for naming prop-holders
     */
    private val attributeNamingComparator: Comparator<Conf> = Comparator
        .comparing { conf: Conf -> conf.isAlreadyBuilt() }.reversed()
        .thenComparingInt { conf -> conf.name.length }
        .thenComparing { conf -> conf.name }

    data class AttributeElement(val className: String, val attributeName: String) {
        constructor(conf: Conf, attribute: AttributeConf) : this(conf.fullName, attribute.name)
    }

    private val attributes: DisjointSetUnion<AttributeElement> = DisjointSetUnionImpl()

    private val propertyHolders: MutableMap<Int, PropHolderMeta> = mutableMapOf()

    init {
        connectAttributes()
        createPropertyHolders()
    }

    fun allPropHolders(): Collection<PropHolderMeta> = propertyHolders.values

    fun propHolderFor(conf: Conf, attribute: AttributeConf): PropHolderMeta? {
        val element = AttributeElement(conf, attribute)
        val propId = attributes.find(element)
            ?: throw IllegalArgumentException("Unknown attribute: $element")
        return propertyHolders[propId]
    }

    /**
     * Add nodes for all attributes and merge connected ones
     */
    private fun connectAttributes() {
        registry.allConfs().forEach { conf ->
            conf.attributes(registry).forEach { attribute ->
                registry.getAllDescendants(conf).forEach { descendant ->
                    attributes.add(AttributeElement(descendant, attribute))
                    attributes.merge(
                        AttributeElement(conf, attribute),
                        AttributeElement(descendant, attribute)
                    )
                }
            }
        }
    }

    /**
     * For each produced attribute set decide if a prop holder is needed, and create one
     */
    private fun createPropertyHolders() {
        val attributeSets: Multimap<Int, AttributeElement> = HashMultimap.create()
        val confByElement: MutableMap<AttributeElement, Conf> = mutableMapOf()

        registry.allConfs().forEach { conf ->
            conf.attributes(registry).forEach { attribute ->
                registry.getAllDescendants(conf).forEach { descendant ->
                    val element = AttributeElement(descendant, attribute)
                    val setId = attributes.find(element)
                    attributeSets.put(setId, element)
                    confByElement[element] = descendant
                }
            }
        }

        for (propId in attributeSets.keySet()) {
            val element: AttributeElement = attributeSets[propId].first()
            val confs: List<Conf> = attributeSets[propId].map { confByElement[it]!! }

            val rootConfs = registry.findRoots(confs)

            val namingClass: Conf = rootConfs.minWithOrNull(attributeNamingComparator)!!

            val attribute: AttributeConf = namingClass.attributes(registry)
                .first { it.name == element.attributeName }

            checkPropHolderIsPossible(rootConfs, element)
            checkAttributeAnnotationsAreTheSame(confs, attribute, namingClass)

            // common root already exists
            if (rootConfs.size == 1) {
                continue
            }

            propertyHolders[propId] = PropHolderMeta(
                name = ClassName.get(
                    "${namingClass.packageName}.prop",
                    "${namingClass.name}${attribute.name.upperCamel()}PropHolder",
                ),
                attribute = attribute,
                isAlreadyBuilt = namingClass.isAlreadyBuilt(),
            )
        }
    }

    private fun checkAttributeAnnotationsAreTheSame(confs: List<Conf>, attribute: AttributeConf, namingClass: Conf) {
        // check that attribute has the same annotations in all confs it is declared in
        confs.forEach { conf ->
            conf.attributes(registry)
                .find { attr -> attr.name == attribute.name }
                ?.let { other ->
                    if (other.annotations.toSet() != attribute.annotations.toSet()) {
                        throw IllegalStateException(
                            "Attribute ${attribute.name} is declared with different set of annotations in " +
                                "class ${namingClass.fullName} and ${conf.fullName}. This is not supported, " +
                                "please either declare attribute in only one conf file (if that is possible), or " +
                                "declare it with the same set of annotations"
                        )
                    }
                }
        }
    }

    private fun checkPropHolderIsPossible(rootConfs: Set<Conf>, element: AttributeElement) {
        // check if common root is needed but can't be created
        val rootModules = rootConfs.groupBy { it.builtModuleName() }
        if (rootModules.size != 1) {
            val exampleClasses = rootModules.values.take(2)
                .joinToString(", ") { it.first().name }
            throw IllegalStateException(
                "Attribute ${element.attributeName} is independently declared in conf files belonging to " +
                    "multiple modules: $exampleClasses. Such attributes require a prop-holder, but prop-holder " +
                    "can't be generated at that stage, as some classes are already generated. Consider creating " +
                    "a common parent class for that attribute."
            )
        }
    }
}
