package ru.yandex.direct.model.generator.rewrite.builder

import com.fasterxml.jackson.annotation.JsonSubTypes
import com.fasterxml.jackson.core.type.TypeReference
import com.squareup.javapoet.AnnotationSpec
import com.squareup.javapoet.ClassName
import com.squareup.javapoet.CodeBlock
import com.squareup.javapoet.FieldSpec
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.ParameterizedTypeName
import com.squareup.javapoet.TypeName
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.model.generator.Tool
import ru.yandex.direct.model.generator.rewrite.conf.AttributeConf
import ru.yandex.direct.model.generator.rewrite.spec.Spec
import ru.yandex.direct.model.generator.rewrite.upperUnderscore
import java.nio.file.Paths
import javax.annotation.Generated
import javax.lang.model.element.Modifier

object BuilderUtils {
    fun generatedAnnotation(spec: Spec): AnnotationSpec =
        AnnotationSpec.builder(Generated::class.java)
            .addMember("value", "\$S", Tool::class.java.canonicalName)
            .apply {
                if (spec.sourceFile != null) {
                    val fileName = Paths.get(spec.sourceFile!!).fileName
                    addMember("comments", "\$S", "generated from $fileName")
                }
            }
            .build()

    fun jsonSubtypesAnnotation(jsonSubtypes: List<TypeName>, jsonSubtypeNames: Boolean): AnnotationSpec {
        val codeBlock: CodeBlock.Builder = CodeBlock.builder()
        jsonSubtypes.withIndex().forEach { (index, subtype) ->
            if (index == 0) {
                codeBlock.add("{\n")
            } else {
                codeBlock.add(",\n")
            }
            if (jsonSubtypeNames) {
                codeBlock.add(
                    "@\$T(value = \$T.class, name = \"\$T\")",
                    JsonSubTypes.Type::class.java, subtype, subtype,
                )
            } else {
                codeBlock.add(
                    "@\$T(\$T.class)",
                    JsonSubTypes.Type::class.java, subtype,
                )
            }
        }
        codeBlock.add("\n}")
        return AnnotationSpec.builder(JsonSubTypes::class.java)
            .addMember("value", codeBlock.build())
            .build()
    }

    fun modelPropertyField(
        spec: Spec,
        attribute: AttributeConf,
        getter: MethodSpec,
        setter: MethodSpec?,
    ): FieldSpec =
        FieldSpec.builder(
            ParameterizedTypeName.get(
                ClassName.get(ModelProperty::class.java),
                spec.name,
                attribute.type.box()
            ),
            attribute.name.upperUnderscore(),
            Modifier.PUBLIC,
            Modifier.STATIC,
            Modifier.FINAL,
        )
            .addJavadoc(attribute.comment ?: "")
            .initializer(
                if (setter != null) {
                    CodeBlock.of(
                        "\n \$1T.create(\$2T.class, \$3S, \$2T::\$4N, \$2T::\$5N)",
                        ClassName.get(ModelProperty::class.java), spec.name, attribute.name, getter, setter,
                    )
                } else {
                    CodeBlock.of(
                        "\n \$1T.createReadOnly(\$2T.class, \$3S, \$2T::\$4N)",
                        ClassName.get(ModelProperty::class.java), spec.name, attribute.name, getter,
                    )
                }
            )
            .build()

    fun allModelPropertiesMethod(attributes: List<AttributeConf>): MethodSpec =
        MethodSpec.methodBuilder("allModelProperties")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
            .returns(
                ParameterizedTypeName.get(
                    Set::class.java,
                    object : TypeReference<ModelProperty<*, *>>() {}.type
                )
            )
            .addStatement(
                "return \$T.of(\n\$L\n)",
                ClassName.get(Set::class.java),
                attributes.joinToString(",\n") { it.name.upperUnderscore() },
            )
            .build()
}
