package ru.yandex.direct.model.generator.rewrite.builder

import com.squareup.javapoet.ClassName
import com.squareup.javapoet.FieldSpec
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.ParameterSpec
import com.squareup.javapoet.TypeName
import com.squareup.javapoet.TypeSpec
import ru.yandex.direct.model.Copyable
import ru.yandex.direct.model.generator.rewrite.conf.Applicability
import ru.yandex.direct.model.generator.rewrite.conf.AttributeConf
import ru.yandex.direct.model.generator.rewrite.spec.ClassSpec
import ru.yandex.direct.model.generator.rewrite.upperCamel
import java.util.Objects
import javax.lang.model.element.Modifier

class ClassBuilder(private val spec: ClassSpec) {
    fun build(): TypeSpec {
        val builder = TypeSpec.classBuilder(spec.name)
            .addAnnotation(BuilderUtils.generatedAnnotation(spec))
            .addJavadoc(spec.comment ?: "")

        spec.annotations
            .forEach { builder.addAnnotation(it.toSpec()) }

        if (spec.jsonSubtypes.isNotEmpty()) {
            builder.addAnnotation(BuilderUtils.jsonSubtypesAnnotation(spec.jsonSubtypes, spec.jsonSubtypeNames))
        }

        spec.modifiers
            .forEach(builder::addModifiers)

        if (spec.extends != null) {
            builder.superclass(spec.extends)
        }

        builder.addSuperinterfaces(spec.implements)

        spec.modelProperties.forEach { attribute ->
            builder.addField(
                BuilderUtils.modelPropertyField(
                    spec,
                    attribute,
                    getterMethod(attribute),
                    setterMethod(attribute)
                )
            )
        }

        spec.inheritedAttributes.forEach { attribute ->
            builder.addMethod(withOverrideSetterMethod(attribute))
        }

        spec.attributes.forEach { attribute ->
            if (attribute.aliasTo == null) {
                builder.addField(attributeField(attribute))
            }

            builder
                .addMethod(getterMethod(attribute))
                .addMethod(setterMethod(attribute))
                .addMethod(withSetterMethod(attribute, setterMethod(attribute)))
        }

        if (spec.generateAllModelProperties) {
            builder.addMethod(allModelPropertiesMethod())
        }

        if (spec.generateCopy) {
            if (!spec.superHasCopy) {
                builder.addSuperinterface(ClassName.get(Copyable::class.java))
            }

            builder.addMethod(copyToMethod())

            if (Modifier.ABSTRACT !in spec.modifiers) {
                builder.addMethod(copyMethod())
            }
        }

        builder
            .addMethod(toStringMethod())
            .addMethod(equalsMethod())
            .addMethod(hashCodeMethod())

        return builder.build()
    }

    private fun withOverrideSetterMethod(attribute: AttributeConf): MethodSpec {
        val setter = setterMethod(attribute)
        return withSetterMethod(attribute, setter).toBuilder()
            .addAnnotation(Override::class.java)
            .build()
    }

    private fun attributeField(attribute: AttributeConf): FieldSpec =
        FieldSpec.builder(attribute.type, attribute.name, Modifier.PRIVATE)
            .apply {
                attribute.annotations
                    .filter { Applicability.FIELD in it.applicability }
                    .forEach { annotation -> addAnnotation(annotation.toSpec()) }
            }
            .build()

    private fun getterMethod(attribute: AttributeConf): MethodSpec =
        MethodSpec.methodBuilder("get${attribute.name.upperCamel()}")
            .addJavadoc(attribute.comment ?: "")
            .addModifiers(Modifier.PUBLIC)
            .returns(attribute.type)
            .addStatement("return \$L", attribute.realName)
            .apply {
                attribute.annotations
                    .filter { Applicability.GETTER in it.applicability }
                    .forEach { annotation -> addAnnotation(annotation.toSpec()) }
            }
            .build()

    private fun setterMethod(attribute: AttributeConf): MethodSpec {
        val parameter = ParameterSpec.builder(attribute.type, attribute.realName)
            .apply {
                attribute.annotations
                    .filter { Applicability.SETTER_PARAMETER in it.applicability }
                    .forEach { annotation -> addAnnotation(annotation.toSpec()) }
            }
            .build()

        return MethodSpec.methodBuilder("set${attribute.name.upperCamel()}")
            .addModifiers(Modifier.PUBLIC)
            .returns(TypeName.VOID)
            .addStatement("this.\$1L = \$1L", attribute.realName)
            .addParameter(parameter)
            .build()
    }

    private fun withSetterMethod(attribute: AttributeConf, setter: MethodSpec): MethodSpec =
        MethodSpec.methodBuilder("with${attribute.name.upperCamel()}")
            .addModifiers(Modifier.PUBLIC)
            .addParameters(setter.parameters)
            .returns(spec.name)
            .addStatement("\$N(\$L)", setter, setter.parameters.first().name)
            .addStatement("return this")
            .build()

    private fun allModelPropertiesMethod(): MethodSpec {
        val allAttributes = (spec.inheritedAttributes + spec.attributes)
            .filter { it.aliasTo == null }
        return BuilderUtils.allModelPropertiesMethod(allAttributes)
    }

    private fun copyToMethod(): MethodSpec {
        val builder = MethodSpec.methodBuilder("copyTo")
            .addModifiers(Modifier.PROTECTED)
            .returns(spec.name)
            .addParameter(spec.name, "target", Modifier.FINAL)

        if (spec.superHasCopy) {
            builder.addStatement("super.copyTo(target)")
        }

        spec.attributes
            .filter { it.aliasTo == null }
            .forEach { attribute ->
                builder.addStatement("target.\$1L = this.\$1L", attribute.name)
            }

        return builder
            .addStatement("return target")
            .build()
    }

    private fun copyMethod(): MethodSpec =
        MethodSpec.methodBuilder("copy")
            .addModifiers(Modifier.PUBLIC)
            .addJavadoc("Create field-for-field copy of this object.\n")
            .addJavadoc("<b>Not deep copy</b>\n\n")
            .addJavadoc("@see #\$L", "copyTo")
            .returns(spec.name)
            .addStatement("final \$1T target = new \$1T()", spec.name)
            .addStatement("return this.copyTo(target)")
            .build()

    private fun toStringMethod(): MethodSpec {
        val methodBuilder: MethodSpec.Builder = MethodSpec.methodBuilder("toString")
            .addModifiers(Modifier.PUBLIC)
            .addAnnotation(Override::class.java)
            .returns(String::class.java)

        methodBuilder
            .addStatement("\$1T sb = new \$1T(\$2S)", StringBuilder::class.java, "${spec.name}{")

        var needLeadingComma = false

        if (spec.extends != null) {
            methodBuilder
                .addStatement("sb.append(\"super=\").append(super.toString())")
            needLeadingComma = true
        }

        spec.attributes
            .filter { it.aliasTo == null }
            .forEach { attribute ->
                if (needLeadingComma) {
                    methodBuilder
                        .addStatement("sb.append(\", \$1L=\").append(\$1N)", attribute.name)
                } else {
                    methodBuilder
                        .addStatement("sb.append(\"\$1L=\").append(\$1N)", attribute.name)
                }
                needLeadingComma = true
            }

        methodBuilder
            .addStatement("sb.append('}')")
            .addStatement("return sb.toString()")

        return methodBuilder.build()
    }

    private fun equalsMethod(): MethodSpec {
        val methodBuilder: MethodSpec.Builder = MethodSpec.methodBuilder("equals")
            .addParameter(Object::class.java, "o")
            .addModifiers(Modifier.PUBLIC)
            .addAnnotation(Override::class.java)
            .returns(TypeName.BOOLEAN)

        methodBuilder
            .beginControlFlow("if (this == o)")
            .addStatement("return true")
            .endControlFlow()
            .beginControlFlow("if (o == null || getClass() != o.getClass())")
            .addStatement("return false")
            .endControlFlow()
            .addStatement("\$1T that = (\$1T) o", spec.name)

        if (spec.extends != null) {
            methodBuilder
                .addStatement("if (!super.equals(o)) \n return false")
        }

        if (spec.attributes.isEmpty()) {
            methodBuilder
                .addStatement("return true")
        } else {
            methodBuilder
                .addStatement(
                    spec.attributes
                        .filter { it.aliasTo == null }
                        .joinToString(prefix = "return ", separator = "\n && ") { attribute ->
                            "\$1T.equals(${attribute.name}, that.${attribute.name})"
                        },
                    Objects::class.java
                )
        }

        return methodBuilder.build()
    }

    private fun hashCodeMethod(): MethodSpec {
        var fields: List<String> = spec.attributes
            .filter { it.aliasTo == null }
            .map { it.name }
            .toMutableList()

        if (spec.extends != null) {
            fields = listOf("super.hashCode()") + fields
        }

        return MethodSpec.methodBuilder("hashCode")
            .addModifiers(Modifier.PUBLIC)
            .addAnnotation(Override::class.java)
            .returns(TypeName.INT)
            .addStatement(
                "return \$T.hash(\n\$L\n)",
                Objects::class.java,
                fields.joinToString(",\n"),
            )
            .build()
    }
}
