package ru.yandex.direct.model.generator.rewrite.builder

import com.fasterxml.jackson.annotation.JsonProperty
import com.squareup.javapoet.AnnotationSpec
import com.squareup.javapoet.ClassName
import com.squareup.javapoet.FieldSpec
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.ParameterSpec
import com.squareup.javapoet.ParameterizedTypeName
import com.squareup.javapoet.TypeName
import com.squareup.javapoet.TypeSpec
import ru.yandex.direct.model.generator.rewrite.spec.EnumSpec
import java.util.Arrays
import java.util.stream.Collectors
import javax.annotation.Nullable
import javax.lang.model.element.Modifier

class EnumBuilder(private val spec: EnumSpec) {
    fun build(): TypeSpec {
        val builder = TypeSpec.enumBuilder(spec.name)
            .addAnnotation(BuilderUtils.generatedAnnotation(spec))
            .addModifiers(Modifier.PUBLIC)
            .addJavadoc(spec.comment ?: "")

        if (spec.valuesSource != null) {
            addImportedValues(builder)
        } else {
            addConstantValues(builder)
            if (spec.valuesType != null) {
                addTypedValues(builder)
            }
        }

        return builder.build()
    }

    private fun addImportedValues(builder: TypeSpec.Builder) {
        val sourceClass = try {
            javaClass.classLoader.loadClass(spec.valuesSource!!.reflectionName())
        } catch (e: ClassNotFoundException) {
            throw IllegalArgumentException("Could not load class ${spec.valuesSource}")
        }

        check(sourceClass.isEnum) {
            "Source class $sourceClass is not a enum"
        }

        val fromSourceBuilder: MethodSpec.Builder = MethodSpec.methodBuilder("fromSource")
            .addParameter(sourceClass, "value")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
            .addAnnotation(Nullable::class.java)
            .returns(spec.name)
            .beginControlFlow("if (value == null)")
            .addStatement("return null")
            .endControlFlow()
            .beginControlFlow("switch(value)")

        val toSourceBuilder: MethodSpec.Builder = MethodSpec.methodBuilder("toSource")
            .addParameter(spec.name, "value")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
            .addAnnotation(Nullable::class.java)
            .returns(sourceClass)
            .beginControlFlow("if (value == null)")
            .addStatement("return null")
            .endControlFlow()
            .beginControlFlow("switch(value)")

        sourceClass.enumConstants.forEach { enumConstant ->
            val source: String = (enumConstant as Enum<*>).name
            val value: String = source.uppercase()

            builder.addEnumConstant(value)
            fromSourceBuilder
                .addStatement("case \$N:\nreturn \$N", source, value)
            toSourceBuilder
                .addStatement("case \$N:\nreturn \$T.\$N", value, sourceClass, source)
        }

        fromSourceBuilder
            .addStatement("default: throw new \$T(\$S + value)", IllegalStateException::class.java, "No such value: ")
            .endControlFlow()

        toSourceBuilder
            .addStatement("default: throw new \$T(\$S + value)", IllegalStateException::class.java, "No such value: ")
            .endControlFlow()

        builder
            .addMethod(fromSourceBuilder.build())
            .addMethod(toSourceBuilder.build())
    }

    private fun addConstantValues(builder: TypeSpec.Builder) {
        spec.values.forEach { value ->
            val typeBuilder = if (value.typedValue == null) {
                TypeSpec.anonymousClassBuilder("")
            } else {
                val format = if (spec.valuesType == TypeName.get(String::class.java)) "\$S" else "\$L"
                TypeSpec.anonymousClassBuilder(format, value.typedValue)
            }

            if (value.comment != null) {
                typeBuilder.addJavadoc(value.comment)
            }

            if (value.jsonProperty != null) {
                typeBuilder.addAnnotation(
                    AnnotationSpec.builder(JsonProperty::class.java)
                        .addMember("value", "\$S", value.jsonProperty)
                        .build()
                )
            }

            builder
                .addEnumConstant(value.value, typeBuilder.build())
        }
    }

    private fun addTypedValues(builder: TypeSpec.Builder) {
        val field = FieldSpec.builder(spec.valuesType, "typedValue")
            .addModifiers(Modifier.PRIVATE, Modifier.FINAL)
            .build()

        val constructor = MethodSpec.constructorBuilder()
            .addParameter(spec.valuesType, field.name)
            .addStatement("this.\$N = \$N", field, field)
            .build()

        val getter = MethodSpec.methodBuilder("getTypedValue")
            .addModifiers(Modifier.PUBLIC)
            .returns(spec.valuesType)
            .addStatement("return \$N", field)
            .build()

        val mapType = ParameterizedTypeName.get(
            ClassName.get(Map::class.java),
            spec.valuesType,
            spec.name,
        )

        val mapField = FieldSpec.builder(mapType, "ENUMS")
            .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
            .initializer(
                "\$T.stream(values())\$W.collect(\$T.toMap(\$T::\$N, e -> e))",
                Arrays::class.java, Collectors::class.java, spec.name, getter,
            )
            .build()

        val fromMethod = MethodSpec.methodBuilder("fromTypedValue")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
            .addParameter(
                ParameterSpec.builder(spec.valuesType, field.name)
                    .addAnnotation(Nullable::class.java)
                    .build()
            )
            .returns(spec.name)
            .beginControlFlow("if (\$N == null)", field)
            .addStatement("return null")
            .endControlFlow()
            .addStatement("\$T result = \$N.get(\$L)", spec.name, mapField, field.name)
            .beginControlFlow("if (result == null)")
            .addStatement(
                "throw new \$T(\$S + \$L)",
                java.lang.IllegalArgumentException::class.java,
                "Unknown enum typed value ", field.name,
            )
            .endControlFlow()
            .addStatement("return result")
            .build()

        builder
            .addField(field)
            .addMethod(constructor)
            .addMethod(getter)
            .addField(mapField)
            .addMethod(fromMethod)
    }
}
