package ru.yandex.direct.model.generator.rewrite.builder

import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.TypeSpec
import ru.yandex.direct.model.generator.rewrite.conf.Applicability
import ru.yandex.direct.model.generator.rewrite.conf.AttributeConf
import ru.yandex.direct.model.generator.rewrite.spec.InterfaceSpec
import ru.yandex.direct.model.generator.rewrite.upperCamel
import javax.lang.model.element.Modifier

class InterfaceBuilder(private val spec: InterfaceSpec) {
    fun build(): TypeSpec {
        val builder = TypeSpec.interfaceBuilder(spec.name)
            .addAnnotation(BuilderUtils.generatedAnnotation(spec))
            .addModifiers(Modifier.PUBLIC)
            .addJavadoc(spec.comment ?: "")

        spec.annotations
            .forEach { builder.addAnnotation(it.toSpec()) }

        if (spec.jsonSubtypes.isNotEmpty()) {
            builder.addAnnotation(BuilderUtils.jsonSubtypesAnnotation(spec.jsonSubtypes, spec.jsonSubtypeNames))
        }

        builder.addSuperinterfaces(spec.extends)

        spec.modelProperties.forEach { attribute ->
            builder.addField(
                BuilderUtils.modelPropertyField(
                    spec,
                    attribute,
                    getterMethod(attribute),
                    setterMethod(attribute)
                        .takeUnless { spec.readonly }
                )
            )
        }

        spec.attributes.forEach { attribute ->
            builder.addMethod(getterMethod(attribute))
            if (!spec.readonly) {
                builder
                    .addMethod(setterMethod(attribute))
                    .addMethod(withSetterMethod(attribute))
            }
        }

        spec.inheritedAttributes.forEach { attribute ->
            if (!spec.readonly) {
                builder.addMethod(withOverrideSetterMethod(attribute))
            }
        }

        if (spec.generateAllModelProperties) {
            builder.addMethod(allModelPropertiesMethod())
        }

        return builder.build()
    }

    private fun getterMethod(attribute: AttributeConf): MethodSpec =
        MethodSpec.methodBuilder("get${attribute.name.upperCamel()}")
            .addJavadoc(attribute.comment ?: "")
            .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
            .returns(attribute.type)
            .apply {
                attribute.annotations
                    .filter { Applicability.GETTER in it.applicability }
                    .forEach { annotation -> addAnnotation(annotation.toSpec()) }
            }
            .build()

    private fun setterMethod(attribute: AttributeConf): MethodSpec =
        MethodSpec.methodBuilder("set${attribute.name.upperCamel()}")
            .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
            .addParameter(attribute.type, attribute.name)
            .build()

    private fun withSetterMethod(attribute: AttributeConf): MethodSpec =
        MethodSpec.methodBuilder("with${attribute.name.upperCamel()}")
            .addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
            .addParameter(attribute.type, attribute.name)
            .returns(spec.name)
            .build()

    private fun withOverrideSetterMethod(attribute: AttributeConf): MethodSpec {
        return withSetterMethod(attribute).toBuilder()
            .addAnnotation(Override::class.java)
            .build()
    }

    private fun allModelPropertiesMethod(): MethodSpec {
        return BuilderUtils.allModelPropertiesMethod(spec.inheritedAttributes + spec.attributes)
    }
}
