package ru.yandex.direct.model.generator.rewrite.builder

import com.squareup.javapoet.ClassName
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.ParameterSpec
import com.squareup.javapoet.ParameterizedTypeName
import com.squareup.javapoet.TypeName
import com.squareup.javapoet.TypeSpec
import ru.yandex.direct.model.Relationship
import ru.yandex.direct.model.generator.rewrite.spec.RelationshipSpec
import ru.yandex.direct.model.generator.rewrite.upperCamel
import javax.annotation.Nullable
import javax.lang.model.element.Modifier

class RelationshipBuilder(private val spec: RelationshipSpec) {
    fun build(): TypeSpec {
        val builder = TypeSpec.classBuilder(spec.name)
            .addAnnotation(BuilderUtils.generatedAnnotation(spec))
            .addJavadoc(spec.comment ?: "")
            .addModifiers(Modifier.PUBLIC)

        builder.addSuperinterface(
            ParameterizedTypeName.get(
                ClassName.get(Relationship::class.java),
                spec.parent,
                spec.child,
                spec.parentIdType,
            )
        )

        return builder
            .addMethod(parentIdGetter())
            .addMethod(parentIdSetter())
            .addMethod(parentEntityGetter())
            .addMethod(childEntityGetter())
            .addMethod(equalsMethod())
            .addMethod(hashCodeMethod())
            .build()
    }

    private fun parentIdGetter(): MethodSpec =
        MethodSpec.methodBuilder("getParentId")
            .addAnnotation(Override::class.java)
            .addAnnotation(Nullable::class.java)
            .addParameter(spec.child, "child")
            .addModifiers(Modifier.PUBLIC)
            .returns(spec.parentIdType)
            .addStatement("return child.\$L()", "get${spec.parentIdField.upperCamel()}").build()

    private fun parentIdSetter(): MethodSpec =
        MethodSpec.methodBuilder("setParentId")
            .addAnnotation(Override::class.java)
            .addParameter(spec.child, "child")
            .addParameter(
                ParameterSpec
                    .builder(spec.parentIdType, "id")
                    .addAnnotation(Nullable::class.java)
                    .build()
            )
            .addModifiers(Modifier.PUBLIC)
            .addStatement("child.\$L(id)", "set${spec.parentIdField.upperCamel()}")
            .build()

    private fun parentEntityGetter(): MethodSpec =
        MethodSpec.methodBuilder("getParentEntityClass")
            .addAnnotation(Override::class.java)
            .addModifiers(Modifier.PUBLIC)
            .returns(ParameterizedTypeName.get(ClassName.get(Class::class.java), spec.parent))
            .addStatement("return \$T.class", spec.parent)
            .build()

    private fun childEntityGetter(): MethodSpec =
        MethodSpec.methodBuilder("getChildEntityClass")
            .addAnnotation(Override::class.java)
            .addModifiers(Modifier.PUBLIC)
            .returns(ParameterizedTypeName.get(ClassName.get(Class::class.java), spec.child))
            .addStatement("return \$T.class", spec.child)
            .build()

    private fun equalsMethod(): MethodSpec =
        MethodSpec.methodBuilder("equals")
            .addAnnotation(Override::class.java)
            .addModifiers(Modifier.PUBLIC)
            .addParameter(Object::class.java, "o")
            .returns(TypeName.BOOLEAN)
            .beginControlFlow("if (this == o)")
            .addStatement("return true")
            .endControlFlow()
            .addStatement("return o != null && getClass() == o.getClass()")
            .build()

    private fun hashCodeMethod(): MethodSpec =
        MethodSpec.methodBuilder("hashCode")
            .addAnnotation(Override::class.java)
            .addModifiers(Modifier.PUBLIC)
            .returns(TypeName.INT)
            .addStatement("return getClass().hashCode()")
            .build()
}
