package ru.yandex.direct.model.generator.rewrite.conf

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.annotation.JsonSubTypes
import com.squareup.javapoet.AnnotationSpec
import com.squareup.javapoet.ClassName
import com.squareup.javapoet.TypeName
import ru.yandex.direct.model.generator.rewrite.ModelRegistry
import ru.yandex.direct.model.Relationship
import javax.lang.model.element.Modifier

/**
 * Config entry declaring a single generated class
 */
interface Conf {
    /**
     * Non-qualified class name
     */
    val name: String

    /**
     * Package name of generated class
     */
    val packageName: String

    /**
     * Comment for generated class
     */
    val comment: String?

    /**
     * Path of config file declaring this class (can be located on classpath)
     */
    val sourceFile: String

    /**
     * Fully-qualified class name of generated class
     */
    val fullName: String
        get() = "$packageName.$name"

    /**
     * List of fully-qualified parent classes and interfaces
     */
    val parents: List<String>
        get() = listOf()
}

/**
 * Common interface for classes which can be annotated with [JsonSubTypes] annotation
 */
interface WithJsonSubtypes : Conf {
    /**
     * If `true`, [JsonSubTypes] annotation will be generated
     */
    val jsonSubtypes: Boolean

    /**
     * If `true`, [JsonSubTypes] annotation will be generated with subtype names, each equal to non-qualified class name
     */
    val jsonSubtypeNames: Boolean
}

/**
 * Config entry corresponding to a single config file (can declare multiple related [Conf] instances)
 */
interface UpperLevelConf : Conf {
    /**
     * List of config entries produced by this config file, including self
     */
    val declaredConfs: List<Conf>
        get() = listOf(this)
}

/**
 * List of attributes declared in [Conf]. Can be attributes of outer [ModelClassConf] for [InnerInterfaceConf]
 */
fun Conf.attributes(registry: ModelRegistry): List<AttributeConf> {
    return when (this) {
        is ModelClassConf -> attributes
        is ModelInterfaceConf -> attributes
        is InnerInterfaceConf -> {
            val classConf = registry.getConf(outerClassName) as ModelClassConf
            val classAttributes = registry.getAllAncestors(classConf)
                // assuming all attributes have to be declared in class confs
                .filterIsInstance<ModelClassConf>()
                .flatMap { it.attributes }
                .distinctBy { it.name }
                .associateBy { it.name }
            return attributes.mapNotNull { classAttributes[it] }
        }
        else -> listOf()
    }
}

/**
 * Checks if the config file declaring this class is located on classpath. Classes produced by such config files are
 * considered to already be generated
 */
fun Conf.isAlreadyBuilt(): Boolean {
    return sourceFile.contains(".jar!/")
}

/**
 * If class produced by this config is already generated, returns jar file path. Otherwise, returns `null`
 *
 * @see isAlreadyBuilt
 */
fun Conf.builtModuleName(): String? {
    if (!isAlreadyBuilt()) {
        return null
    }
    val (module, _) = sourceFile.split("!", limit = 2)
    return module
}

// =====================================================================================================================

/**
 * Specifies where annotation should be generated
 */
enum class Applicability {
    /**
     * Only annotate field (has no effect in interfaces)
     */
    FIELD,

    /**
     * Annotate getter method
     */
    GETTER,

    /**
     * Annotate setter parameter
     */
    SETTER_PARAMETER,
}

/**
 * Specifies a single annotation parameter
 */
data class AnnotationParameterConf(
    /**
     * Name of parameter to be set
     */
    val key: String,
    /**
     * Parameter value formatter (using java-poet format)
     */
    val formatter: String,
    /**
     * Parameter value template argument (to be used in [formatter])
     */
    val codeBlock: String,
)

/**
 * Specifies a single annotation (on attribute or a class)
 */
data class AnnotationConf(
    /**
     * Annotation type [ClassName]
     */
    val className: ClassName,
    /**
     * Where the annotation should be generated (has no effect on class/interface annotations)
     */
    val applicability: Set<Applicability>,
    /**
     * Annotation parameters to be generated
     */
    val parameters: List<AnnotationParameterConf>,
) {
    /**
     * Builds [AnnotationSpec] for this annotation
     */
    fun toSpec(): AnnotationSpec {
        return AnnotationSpec.builder(className)
            .apply { parameters.forEach { param -> addMember(param.key, param.formatter, param.codeBlock) } }
            .build()
    }
}

/**
 * Specifies a single interface/class attribute
 */
data class AttributeConf(
    /**
     * Attribute name
     */
    val name: String,
    /**
     * Attribute type
     */
    val type: TypeName,
    /**
     * Comment for attribute
     */
    val comment: String? = null,
    /**
     * If non-null, field will not be generated. Getters and setters for this attribute will reference field [aliasTo]
     */
    val aliasTo: String? = null,
    /**
     * If non-null, [JsonProperty] annotation will be generated, with value equal to [jsonProperty]
     */
    val jsonProperty: String? = null,
    /**
     * Is non-null, [JsonInclude] annotation will be generated, with value set to [jsonInclude]
     * @see JsonInclude.Include
     */
    val jsonInclude: String? = null,
    /**
     * Annotations declared in config file
     */
    val configAnnotations: List<AnnotationConf> = listOf(),
    /**
     * If non-null, [Relationship] referencing this attribute will be generated
     */
    val relationship: RelationshipConf? = null,
) {
    /**
     * Name of the field, which this attribute is referencing
     */
    val realName: String = aliasTo ?: name

    /**
     * Full list of annotations for this attribute
     */
    val annotations: List<AnnotationConf> = configAnnotations +
        (if (jsonProperty != null) listOf(jsonPropertyAnnotation(jsonProperty)) else listOf()) +
        (if (jsonInclude != null) listOf(jsonIncludeAnnotation(jsonInclude)) else listOf())

    private fun jsonPropertyAnnotation(jsonProperty: String) = AnnotationConf(
        className = ClassName.get(JsonProperty::class.java),
        applicability = setOf(Applicability.FIELD),
        parameters = listOf(
            AnnotationParameterConf(
                "value", "\$S", jsonProperty,
            )
        ),
    )

    private fun jsonIncludeAnnotation(jsonInclude: String) = AnnotationConf(
        className = ClassName.get(JsonInclude::class.java),
        applicability = setOf(Applicability.FIELD),
        parameters = listOf(
            AnnotationParameterConf(
                "value", "\$L", "JsonInclude.Include.$jsonInclude",
            )
        ),
    )
}

/**
 * Specifies single enum constant
 */
data class EnumValueConf(
    /**
     * Enum constant name
     */
    val value: String?,
    /**
     * Enum constant comment
     */
    val comment: String? = null,
    /**
     * Typed value (with type specified by [EnumConf.valuesType]), contained in enum constant.
     *
     * For example, `ENUM_VALUE(123)` will be generated for [typedValue] equal to `"123"` if [EnumConf.valuesType] is
     * set to `"Long"`
     */
    val typedValue: String? = null,
    /**
     * If non-null, [JsonProperty] annotation will be generated, with value equal to [jsonProperty]
     */
    val jsonProperty: String? = null,
)

// =====================================================================================================================

/**
 * Common interface for [ModelInterfaceConf] and [InnerInterfaceConf]
 */
interface InterfaceConf : Conf, WithJsonSubtypes {
    /**
     * Annotations to be generated on this interface
     */
    val annotations: List<AnnotationConf>

    /**
     * If `true`, only getters will be generated for attributes of the interface
     */
    val readonly: Boolean

    /**
     * List of fully-qualified extended interface names
     */
    val extends: List<String>
}

// =====================================================================================================================

/**
 * Specifies a single class (may contain related interfaces and enums)
 */
data class ModelClassConf(
    override val name: String,
    override val packageName: String,
    override val comment: String?,
    override val sourceFile: String,

    /**
     * Annotations to be generated on this class
     */
    val annotations: List<AnnotationConf>,
    override val jsonSubtypes: Boolean,
    override val jsonSubtypeNames: Boolean,

    /**
     * Modifiers to be added to generated class
     */
    val modifiers: List<Modifier>,

    /**
     * Optional fully-qualified parent class name
     */
    val extends: String?,
    /**
     * List of fully-qualified implemented interface names
     */
    val implements: List<String>,

    /**
     * Attributes of the class
     */
    val attributes: List<AttributeConf>,

    /**
     * If true, copy method will be generated
     */
    val generateCopy: Boolean,
    /**
     * If true, static ModelProperty instances will be generated
     */
    val generateProperties: Boolean,

    /**
     * Related enum classes, declared in the same config file
     */
    val enums: List<EnumConf>,
    /**
     * Related interfaces, declared in the same config file
     */
    val interfaces: List<InnerInterfaceConf>,
) : UpperLevelConf, WithJsonSubtypes {
    override val declaredConfs: List<Conf> =
        listOf(this) + enums + interfaces +
            attributes.mapNotNull { it.relationship }

    override val parents: List<String> =
        implements +
            interfaces.map { it.fullName } +
            extends?.let { listOf(it) }.orEmpty()
}

/**
 * Specifies an interface with a subset of class [outerClassName] fields, which optionally can be readonly
 */
data class InnerInterfaceConf(
    override val name: String,
    override val packageName: String,
    override val comment: String?,
    override val sourceFile: String,

    override val annotations: List<AnnotationConf>,
    override val jsonSubtypes: Boolean,
    override val jsonSubtypeNames: Boolean,

    /**
     * Attributes of the interface
     */
    val attributes: List<String>,
    override val readonly: Boolean,

    override val extends: List<String>,

    /**
     * Fully-qualified name of a class declaring this interface
     */
    val outerClassName: String,
) : InterfaceConf {
    override val parents: List<String> = extends
}

/**
 * Specifies a single interface (may contain related enums)
 */
data class ModelInterfaceConf(
    override val name: String,
    override val packageName: String,
    override val comment: String?,
    override val sourceFile: String,

    override val annotations: List<AnnotationConf>,
    override val jsonSubtypes: Boolean,
    override val jsonSubtypeNames: Boolean,

    /**
     * Attributes of the interface
     */
    val attributes: List<AttributeConf>,
    override val readonly: Boolean,

    override val extends: List<String>,

    /**
     * If true, static ModelProperty instances will be generated
     */
    val generateProperties: Boolean,

    /**
     * Related enum classes, declared in the same config file
     */
    val enums: List<EnumConf>,
) : UpperLevelConf, InterfaceConf {
    override val declaredConfs: List<Conf> =
        listOf(this) + enums +
            attributes.mapNotNull { it.relationship }

    override val parents: List<String> = extends
}

/**
 * Specifies a single relationship class
 */
data class RelationshipConf(
    override val name: String,
    override val packageName: String,
    override val comment: String?,
    override val sourceFile: String,

    /**
     * Fully-qualified parent class name
     */
    val parent: String,
    /**
     * Fully-qualified child class name
     */
    val child: String,

    /**
     * Parent id field name
     */
    val parentIdField: String,
    /**
     * Parent id field type
     */
    val parentIdType: String,
) : UpperLevelConf

/**
 * Specifies a single enum class
 */
data class EnumConf(
    override val name: String,
    override val packageName: String,
    override val comment: String?,
    override val sourceFile: String,

    /**
     * If non-null, type of enum typed value field
     */
    val valuesType: String?,
    /**
     * If non-null, fully-qualified name of enum class, which will be the source of enum constants
     */
    val valuesSource: String?,
    /**
     * List of enum constants
     */
    val values: List<EnumValueConf>,
) : UpperLevelConf {
    init {
        check(values.isEmpty() || valuesSource == null) {
            "Enum should either have its own values or valuesSource, not both"
        }
    }
}
