package ru.yandex.direct.model.generator.rewrite.conf

import com.typesafe.config.Config
import com.typesafe.config.ConfigFactory
import com.typesafe.config.ConfigObject
import com.typesafe.config.ConfigValue
import org.apache.commons.lang3.StringEscapeUtils
import ru.yandex.direct.model.generator.rewrite.JavaPoetUtils
import java.net.URL
import java.nio.file.Paths
import javax.lang.model.element.Modifier
import kotlin.io.path.name

class ConfParser {
    fun parseConfFile(source: URL): UpperLevelConf {
        val config = ConfigFactory.parseURL(source)
        val sourceFileName = Paths.get(source.file).name
        return when {
            sourceFileName.startsWith("i_") -> createModelInterfaceConf(config, source.file)
            sourceFileName.startsWith("r_") -> createRelationshipConf(config, source.file)
            sourceFileName.startsWith("e_") -> createEnumConf(config, source.file)
            else -> createModelClassConf(config, source.file)
        }
    }

    private fun createModelClassConf(config: Config, sourceFileName: String): ModelClassConf {
        val name = config.getString("name")
        val packageName = config.getString("package")
        return ModelClassConf(
            name = name,
            packageName = packageName,
            comment = config.getStringOrNull("comment"),
            sourceFile = sourceFileName,

            annotations = config.getConfigListOrEmpty("annotated_by")
                .map { annotation -> createAnnotationConfig(annotation, packageName) },
            jsonSubtypes = config.getBooleanOrFalse("jsonSubtypes"),
            jsonSubtypeNames = config.getBooleanOrFalse("jsonSubtypesWithNameValue"),

            modifiers = config.getStringListOrEmpty("modifiers")
                .map { modifier -> Modifier.valueOf(modifier) }
                .ifEmpty { listOf(Modifier.PUBLIC) },

            extends = config.getStringOrNull("extends")
                ?.let { JavaPoetUtils.toFullName(it, packageName) },
            implements = config.getStringListOrEmpty("implements")
                .map { JavaPoetUtils.toFullName(it, packageName) },

            attributes = config.getConfigListOrEmpty("attrs")
                .map { attribute -> createAttributeConfig(attribute, name, packageName, sourceFileName) },

            generateCopy = config.getBooleanOrFalse("generateCopyMethod"),
            generateProperties = config.getBooleanOrTrue("generateProperties"),

            enums = config.getConfigListOrEmpty("enums")
                .map { enum -> createEnumConf(enum, sourceFileName, packageName) },
            interfaces = config.getConfigListOrEmpty("interfaces")
                .map { interfaceConfig ->
                    createInnerInterfaceConf(
                        interfaceConfig,
                        packageName,
                        name,
                        sourceFileName
                    )
                },
        )
    }

    private fun createInnerInterfaceConf(
        config: Config,
        packageName: String,
        outerClassName: String,
        sourceFileName: String,
    ): InnerInterfaceConf {
        return InnerInterfaceConf(
            name = config.getString("name"),
            packageName = packageName,
            comment = config.getStringOrNull("comment"),
            sourceFile = sourceFileName,

            annotations = config.getConfigListOrEmpty("annotated_by")
                .map { annotation -> createAnnotationConfig(annotation, packageName) },
            jsonSubtypes = config.getBooleanOrFalse("jsonSubtypes"),
            jsonSubtypeNames = config.getBooleanOrFalse("jsonSubtypesWithNameValue"),

            attributes = config.getStringListOrEmpty("attrs"),
            readonly = config.getBooleanOrFalse("readonly"),

            extends = config.getStringListOrEmpty("extends")
                .map { JavaPoetUtils.toFullName(it, packageName) },

            outerClassName = JavaPoetUtils.toFullName(outerClassName, packageName),
        )
    }

    private fun createModelInterfaceConf(config: Config, sourceFileName: String): ModelInterfaceConf {
        val name = config.getString("name")
        val packageName = config.getString("package")
        return ModelInterfaceConf(
            name = name,
            packageName = packageName,
            comment = config.getStringOrNull("comment"),
            sourceFile = sourceFileName,

            annotations = config.getConfigListOrEmpty("annotated_by")
                .map { annotation -> createAnnotationConfig(annotation, packageName) },
            jsonSubtypes = config.getBooleanOrFalse("jsonSubtypes"),
            jsonSubtypeNames = config.getBooleanOrFalse("jsonSubtypesWithNameValue"),

            attributes = config.getConfigListOrEmpty("attrs")
                .map { attribute -> createAttributeConfig(attribute, name, packageName, sourceFileName) },
            readonly = config.getBooleanOrFalse("readonly"),

            extends = config.getStringListOrEmpty("extends")
                .map { JavaPoetUtils.toFullName(it, packageName) },

            generateProperties = config.getBooleanOrTrue("generateProperties"),

            enums = config.getConfigListOrEmpty("enums")
                .map { enum -> createEnumConf(enum, sourceFileName, packageName) },
        )
    }

    private fun createAttributeConfig(
        config: Config,
        className: String,
        packageName: String,
        sourceFileName: String,
    ): AttributeConf {
        val name = config.getString("name")
        val type = config.getString("type")
        return AttributeConf(
            name = name,
            type = JavaPoetUtils.toTypeName(type, packageName),
            comment = config.getStringOrNull("comment"),
            aliasTo = config.getStringOrNull("aliasTo"),
            jsonProperty = config.getStringOrNull("json"),
            jsonInclude = config.getStringOrNull("jsonInclude"),
            configAnnotations = config.getConfigListOrEmpty("annotated_by")
                .map { annotation -> createAnnotationConfig(annotation, packageName) },
            relationship = config.getConfigOrNull("relationship")
                ?.let { createAttributeRelationshipConf(it, name, type, className, packageName, sourceFileName) },
        )
    }

    private fun createAttributeRelationshipConf(
        config: Config,
        attributeName: String,
        attributeType: String,
        className: String,
        packageName: String,
        sourceFileName: String,
    ): RelationshipConf {
        return RelationshipConf(
            name = config.getString("name"),
            packageName = packageName,
            comment = null,
            sourceFile = sourceFileName,

            parent = JavaPoetUtils.toFullName(config.getString("parent"), packageName),
            child = JavaPoetUtils.toFullName(className, packageName),

            parentIdField = attributeName,
            parentIdType = attributeType,
        )
    }

    private fun createRelationshipConf(config: Config, sourceFileName: String): RelationshipConf {
        val packageName = config.getString("package")
        return RelationshipConf(
            name = config.getString("name"),
            packageName = packageName,
            comment = null,
            sourceFile = sourceFileName,

            parent = JavaPoetUtils.toFullName(config.getString("parent"), packageName),
            child = JavaPoetUtils.toFullName(config.getString("child"), packageName),

            parentIdField = config.getString("parentIdField"),
            parentIdType = config.getStringOrNull("parentIdType") ?: "Long",
        )
    }

    private fun createEnumConf(config: Config, sourceFileName: String, defaultPackage: String? = null): EnumConf {
        return EnumConf(
            name = config.getString("name"),
            packageName = config.getStringOrNull("package")
                ?: defaultPackage
                ?: throw IllegalArgumentException("No enum package"),
            comment = config.getStringOrNull("comment"),
            sourceFile = sourceFileName,

            valuesType = config.getStringOrNull("valuesType"),
            values = config.getListOrEmpty("values")
                .map { value -> createEnumValueConf(value) },

            valuesSource = config.getStringOrNull("valuesSource"),
        )
    }

    private fun createEnumValueConf(config: ConfigValue): EnumValueConf {
        return when (config) {
            is ConfigObject -> {
                val asConfig = config.toConfig()
                EnumValueConf(
                    value = asConfig.getString("value"),
                    comment = asConfig.getStringOrNull("comment"),
                    typedValue = asConfig.getStringOrNull("typedValue"),
                    jsonProperty = asConfig.getStringOrNull("jsonProperty"),
                )
            }
            else -> EnumValueConf(
                value = config.unwrapped().toString(),
            )
        }
    }

    private fun createAnnotationConfig(config: Config, packageName: String?): AnnotationConf {
        val applicability = config.getStringListOrNull("apply")
            ?.map { Applicability.valueOf(it.uppercase()) }
            ?.toSet()
            ?: setOf(Applicability.FIELD, Applicability.GETTER)

        val parameter = config.getConfigListOrEmpty("params")
            .map { param -> createParameterConfig(param) }

        return AnnotationConf(
            className = JavaPoetUtils.toClassName(config.getString("type"), packageName),
            applicability = applicability,
            parameters = parameter,
        )
    }

    private fun createParameterConfig(config: Config): AnnotationParameterConf {
        val isLiteral = config.getBooleanOrFalse("literal")
        val value: Any? = config.getAnyRef("value")
        return AnnotationParameterConf(
            key = config.getString("key"),
            formatter = when (value) {
                is Long, is Int, is Boolean -> "\$L"
                is String -> if (isLiteral) "\$L" else "\$S"
                is List<*> -> "\$L"
                else -> throw IllegalArgumentException("Invalid annotation parameter value: $value")
            },
            codeBlock = when (value) {
                is Long, is Int, is Boolean -> value.toString()
                is String -> value
                is List<*> -> value.joinToString(",", "{", "}") {
                    if (!isLiteral && it is String) {
                        """"${StringEscapeUtils.escapeJava(it)}""""
                    } else {
                        it.toString()
                    }
                }
                else -> throw IllegalArgumentException("Invalid annotation parameter value: $value")
            },
        )
    }
}

