package ru.yandex.direct.model.generator.rewrite.spec

import com.squareup.javapoet.ClassName
import com.squareup.javapoet.TypeName
import ru.yandex.direct.model.generator.rewrite.JavaPoetUtils
import ru.yandex.direct.model.generator.rewrite.ModelRegistry
import ru.yandex.direct.model.generator.rewrite.PropertyResolver
import ru.yandex.direct.model.generator.rewrite.conf.AttributeConf
import ru.yandex.direct.model.generator.rewrite.conf.Conf
import ru.yandex.direct.model.generator.rewrite.conf.ModelClassConf
import ru.yandex.direct.model.generator.rewrite.conf.attributes

class ClassSpecFactory(
    private val registry: ModelRegistry,
    private val propertyResolver: PropertyResolver,
) {
    fun createClassSpec(conf: ModelClassConf): ClassSpec {
        val attributes = conf.attributes
        val attributeNames = attributes.map { it.name }.toSet()

        // Attributes from parent class have priority over attributes from interfaces
        val extendsConfs: Set<Conf> = conf.extends
            ?.let { registry.getConf(it) }
            ?.let { registry.getAllAncestors(it) }
            .orEmpty()
        val implementsConfs: Set<Conf> = registry.getAllAncestors(conf).minus(conf)
            .minus(extendsConfs)

        // Names of the attributes that are present in any of the parent classes
        val superAttributes = (extendsConfs + implementsConfs)
            .flatMap { it.attributes(registry) }
            .distinctBy { it.name }
        val superAttributeNames = superAttributes.map { it.name }.toSet()

        // Attributes that are not declared in this conf, but are present in any of the parent classes
        val inheritedAttributes = superAttributes
            .filter { it.name !in attributeNames }

        val modelProperties: List<AttributeConf> = attributes
            .filter { conf.generateProperties }
            .filter { it.name !in superAttributeNames }
            .filter { propertyResolver.propHolderFor(conf, it) == null }

        val implements: MutableList<TypeName> = conf.interfaces
            .map { ClassName.get(it.packageName, it.name) }
            .toMutableList()

        implements += conf.implements.distinct()
            .map { JavaPoetUtils.toTypeName(it, conf.packageName) }

        implements += conf.attributes
            .filter { it.name !in superAttributeNames }
            .mapNotNull { propertyResolver.propHolderFor(conf, it) }
            .map { it.name }

        val needsModelInterface = registry.getAllAncestors(conf).minus(conf).isEmpty()
            || (!SpecUtils.hasIdAttribute(superAttributes) && SpecUtils.hasIdAttribute(attributes))
        if (conf.generateProperties && needsModelInterface) {
            implements += SpecUtils.modelInterface(attributes + inheritedAttributes)
        }

        val superHasCopyMethod = registry.getAllAncestors(conf).minus(conf)
            .filterIsInstance<ModelClassConf>()
            .any { it.generateCopy }

        return ClassSpec(
            name = ClassName.get(conf.packageName, conf.name),
            comment = conf.comment,
            sourceFile = conf.sourceFile,

            annotations = conf.annotations,

            attributes = attributes,
            inheritedAttributes = inheritedAttributes,
            modelProperties = modelProperties,

            extends = conf.extends?.let { JavaPoetUtils.toClassName(it) },
            implements = implements,

            modifiers = conf.modifiers,

            jsonSubtypes = SpecUtils.jsonSubtypes(conf, registry),
            jsonSubtypeNames = conf.jsonSubtypeNames,

            generateCopy = conf.generateCopy || superHasCopyMethod,
            superHasCopy = superHasCopyMethod,

            generateAllModelProperties = conf.generateProperties,
        )
    }
}
