package ru.yandex.direct.model.generator.rewrite.spec

import com.squareup.javapoet.ClassName
import ru.yandex.direct.model.generator.rewrite.JavaPoetUtils
import ru.yandex.direct.model.generator.rewrite.ModelRegistry
import ru.yandex.direct.model.generator.rewrite.PropHolderMeta
import ru.yandex.direct.model.generator.rewrite.PropertyResolver
import ru.yandex.direct.model.generator.rewrite.conf.InnerInterfaceConf
import ru.yandex.direct.model.generator.rewrite.conf.InterfaceConf
import ru.yandex.direct.model.generator.rewrite.conf.ModelClassConf
import ru.yandex.direct.model.generator.rewrite.conf.ModelInterfaceConf
import ru.yandex.direct.model.generator.rewrite.conf.attributes
import java.nio.file.Paths
import kotlin.io.path.name

class InterfaceSpecFactory(
    private val registry: ModelRegistry,
    private val propertyResolver: PropertyResolver,
) {
    fun createInterfaceSpec(conf: ModelInterfaceConf): InterfaceSpec {
        return createInterfaceSpec(conf, conf.generateProperties)
    }

    fun createInnerInterfaceSpec(conf: InnerInterfaceConf): InterfaceSpec {
        val classConf = registry.getConf(conf.outerClassName) as ModelClassConf
        return createInterfaceSpec(conf, classConf.generateProperties)
    }

    private fun createInterfaceSpec(
        conf: InterfaceConf,
        generateProperties: Boolean,
    ): InterfaceSpec {
        // Attributes that are present in any of the parent classes
        val superAttributes = registry.getAllAncestors(conf).minus(conf)
            .flatMap { subConf -> subConf.attributes(registry) }
            .distinctBy { it.name }
        val superAttributeNames = superAttributes.map { it.name }.toSet()

        checkAttributeDeclarationsAreNotDuplicated(conf, superAttributeNames)

        // Attributes that are first appearing in this class
        val attributes = conf.attributes(registry)
            .filter { it.name !in superAttributeNames }
        val (selfAttributes, propHolderAttributes) = attributes
            .partition { propertyResolver.propHolderFor(conf, it) == null }

        val extends = conf.extends
            .map { JavaPoetUtils.toTypeName(it) }
            .toMutableList()
        extends += propHolderAttributes
            .mapNotNull { propertyResolver.propHolderFor(conf, it) }
            .map { it.name }

        val needsModelInterface = registry.getAllAncestors(conf).minus(conf).isEmpty()
            || (!SpecUtils.hasIdAttribute(superAttributes) && SpecUtils.hasIdAttribute(attributes))
        if (needsModelInterface) {
            extends += SpecUtils.modelInterface(attributes + superAttributes)
        }

        return InterfaceSpec(
            name = ClassName.get(conf.packageName, conf.name),
            comment = conf.comment,
            sourceFile = conf.sourceFile,

            annotations = conf.annotations,

            attributes = selfAttributes,
            inheritedAttributes = (propHolderAttributes + superAttributes),
            modelProperties = selfAttributes
                .filter { generateProperties },

            extends = extends,

            jsonSubtypes = SpecUtils.jsonSubtypes(conf, registry),
            jsonSubtypeNames = conf.jsonSubtypeNames,

            readonly = conf.readonly,
            generateAllModelProperties = generateProperties,
        )
    }

    fun createPropHolderInterfaceSpec(propHolderMeta: PropHolderMeta): InterfaceSpec {
        return InterfaceSpec(
            name = propHolderMeta.name,
            comment = "Auxiliary generated interface. Please do NOT reference it directly!\n",
            sourceFile = null,

            annotations = listOf(),

            attributes = listOf(propHolderMeta.attribute),
            inheritedAttributes = listOf(),
            modelProperties = listOf(propHolderMeta.attribute),

            extends = listOf(SpecUtils.modelInterface(listOf(propHolderMeta.attribute))),

            jsonSubtypes = listOf(),
            jsonSubtypeNames = false,

            readonly = false,
            generateAllModelProperties = false,
        )
    }

    private fun checkAttributeDeclarationsAreNotDuplicated(
        conf: InterfaceConf,
        superAttributeNames: Set<String>,
    ) {
        val duplicatedAttributes = conf.attributes(registry)
            .filter { it.name in superAttributeNames }
            .map { it.name }
        if (duplicatedAttributes.isNotEmpty()) {
            throw IllegalStateException(
                "Attributes $duplicatedAttributes are already declared in parent classes of ${conf.name}, " +
                    "remove them from ${Paths.get(conf.sourceFile).name}"
            )
        }
    }
}
