package ru.yandex.direct.model.generator.uf;

import java.util.Arrays;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Оптимизация {@link QuickUnion} взвешиванием множеств.
 * <p>
 * При объединении меньшие множества вливаются в большие,
 * тем самым балансируя деревья.
 * Глубина дерева не превышает log2(N).
 */
@ParametersAreNonnullByDefault
class WeightedQuickUnion<E> extends QuickUnion<E> {

    private final int[] size;

    WeightedQuickUnion(Collection<E> items) {
        super(items);
        size = new int[items.size()];
        Arrays.fill(size, 1);
    }

    @Override
    public void connect(E p, E q) {
        E rootP = root(p);
        E rootQ = root(q);
        if (rootP == rootQ) {
            return;
        }
        if (sizeOfSet(rootP) > sizeOfSet(rootQ)) {
            mergeFirstToSecond(rootQ, rootP);
            increaseSetSizeBy(rootP, sizeOfSet(rootQ));
        } else {
            mergeFirstToSecond(rootP, rootQ);
            increaseSetSizeBy(rootQ, sizeOfSet(rootP));
        }
    }

    private int sizeOfSet(E root) {
        return size[index.get(root)];
    }

    private void increaseSetSizeBy(E root, int increment) {
        size[index.get(root)] += increment;
    }

}
