package ru.yandex.direct.model;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.utils.CollectionUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableMap;
import static java.util.Collections.unmodifiableSet;

/**
 * Позволяет изменять объект модели, сохраняя при этом оригинальные значения свойств и перечень изменённых объектов.
 * Полезно в ситуациях, когда требуется минимизировать количество обновляемых значений в БД,
 * или реализовать триггер, срабатывающий на изменение свойств объекта модели.
 * <p>
 * Экземпляр {@link AppliedChanges} хранит в себе объект модели и позволяет модифицировать его через
 * метод {@link #modify(ModelProperty, Object)}. Если изменение не меняет свойства, оно игнорируется.
 * Список {@link ModelProperty}, которые были изменены можно получить вызовом {@link #getActuallyChangedProps()}.
 * Узнать, менялось ли конкретное свойство можно вызовом {@link #changed(ModelProperty)}.
 * {@link #getOldValue(ModelProperty)} возвращает оригинальное значение объекта модели до изменений.
 */
@ParametersAreNonnullByDefault
public class AppliedChanges<M extends Model> {
    private final M model;
    private final Map<ModelProperty<? super M, ?>, Object> oldValues;
    private final Set<ModelProperty<? super M, ?>> sensitiveProperties;
    private final Set<ModelProperty<? super M, ?>> affectedSensitiveProperties;
    private final Set<ModelProperty<? super M, ?>> passedProperties;
    private final Map<ModelProperty<? super M, ?>, BiFunction<Object, Object, Boolean>> customModelEquals;

    AppliedChanges(M model) {
        this(model, emptySet());
    }

    AppliedChanges(M model, Set<ModelProperty<? super M, ?>> sensitiveProperties) {
        this(model, sensitiveProperties, emptyMap());
    }

    AppliedChanges(M model, Set<ModelProperty<? super M, ?>> sensitiveProperties,
                   Map<ModelProperty<? super M, ?>, BiFunction<Object, Object, Boolean>> customModelEquals) {
        this.model = checkNotNull(model, "Can't create AppliedChanges instance on null model");
        this.sensitiveProperties = unmodifiableSet(sensitiveProperties);
        this.oldValues = new HashMap<>();
        this.affectedSensitiveProperties = new HashSet<>();
        this.customModelEquals = unmodifiableMap(customModelEquals);
        this.passedProperties = new HashSet<>();
    }

    @SuppressWarnings("CheckReturnValue")
    public AppliedChanges(AppliedChanges<M> appliedChanges, Set<ModelProperty<? super M, ?>> sensitiveProperties,
                          Map<ModelProperty<? super M, ?>, BiFunction<Object, Object, Boolean>> customModelEquals) {
        this.sensitiveProperties = sensitiveProperties;
        checkNotNull(appliedChanges, "Can't create AppliedChanges instance on null");
        this.model = appliedChanges.model;
        this.oldValues = appliedChanges.oldValues;
        this.affectedSensitiveProperties = new HashSet<>();
        this.customModelEquals = unmodifiableMap(customModelEquals);
        this.passedProperties = new HashSet<>();
    }


    public static <X extends Model, V> Predicate<AppliedChanges<X>> isChanged(
            ModelProperty<? super X, V> modelProperty) {
        return xAppliedChanges -> xAppliedChanges.changed(modelProperty);
    }

    public static <X extends Model, V> Predicate<AppliedChanges<X>> isChangedTo(
            ModelProperty<? super X, V> modelProperty, V newValue) {
        return xAppliedChanges -> xAppliedChanges.changed(modelProperty) &&
                xAppliedChanges.customEquals(modelProperty, xAppliedChanges.getNewValue(modelProperty), newValue);
    }

    /**
     * Аналог метода {@link #isChanged(ModelProperty)}
     * с подсказкой компилятору в виде указателя на класс – для явного определения типа
     * в местах, где другого способа нет.
     *
     * @param modelType игнорируется
     */
    public static <X extends Model, V> Predicate<AppliedChanges<X>> isChanged(
            ModelProperty<? super X, V> modelProperty, @SuppressWarnings("unused") Class<X> modelType) {
        return xAppliedChanges -> xAppliedChanges.changed(modelProperty);
    }

    public static <X extends Model, V> Consumer<AppliedChanges<X>> setter(ModelProperty<? super X, V> modelProperty,
                                                                          @Nullable V value) {
        return xAppliedChanges -> xAppliedChanges.modify(modelProperty, value);
    }

    public static <X extends Model, V> Consumer<AppliedChanges<X>> mapper(
            ModelProperty<? super X, V> modelProperty,
            Function<V, V> valueMapper) {
        return xAppliedChanges -> xAppliedChanges.modify(modelProperty,
                valueMapper.apply(xAppliedChanges.getNewValue(modelProperty)));
    }

    /**
     * Применяет изменение к объекту. Если новое значение равно оригинальному, изменение игнорируется.
     *
     * @param changedProp изменяемое свойство
     * @param newValue    новое значение
     */
    public <V> void modify(ModelProperty<? super M, V> changedProp, @Nullable V newValue) {
        passedProperties.add(changedProp);
        // если было изменено чувствительное значение вроде флагов модерации - записываем
        if (sensitiveProperties.contains(changedProp)) {
            affectedSensitiveProperties.add(changedProp);
        }

        boolean valueAlreadyChanged = oldValues.containsKey(changedProp);
        // если значение уже было изменено, то проверяем, не равно ли новое значение первоначальному

        if (valueAlreadyChanged) {
            //noinspection unchecked
            V oldValue = (V) oldValues.get(changedProp);
            boolean equalToOldValue = customEquals(changedProp, oldValue, newValue);
            if (equalToOldValue) {
                changedProp.set(model, oldValue);
                oldValues.remove(changedProp);
                return;
            }
        }

        // если значение еще не менялось, либо менялось, но новое не равно первоначальному,
        // то проверяем, не равно ли новое значение текущему
        Object currentValue = changedProp.get(model);
        boolean equalToCurrentValue = customEquals(changedProp, currentValue, newValue);
        if (!equalToCurrentValue) {
            changedProp.set(model, newValue);
            // на случай, если значение уже было изменено, чтобы не потерять первоначальное значение
            if (!oldValues.containsKey(changedProp)) {  // не используем putIfAbsent, т.к. теряет null-значения
                oldValues.put(changedProp, currentValue);
            }
        }
    }

    private <V> boolean customEquals(ModelProperty<? super M, V> property,
                                     @Nullable Object value1, @Nullable Object value2) {
        BiFunction<Object, Object, Boolean> equalsFunctionForProperty =
                customModelEquals.getOrDefault(property, this::defaultEquals);
        return equalsFunctionForProperty.apply(value1, value2);
    }

    private boolean defaultEquals(@Nullable Object obj1, @Nullable Object obj2) {
        boolean valuesEqual = Objects.equals(obj1, obj2);

        // если значения не равны, но являются BigDecimal, сравниваем через compareTo
        if (!valuesEqual &&
                BigDecimal.class.isInstance(obj1) &&
                BigDecimal.class.isInstance(obj2)) {
            // isInstance попутно проверяет на null
            int comparisonResult = ((BigDecimal) obj1).compareTo((BigDecimal) obj2);
            valuesEqual = comparisonResult == 0;
        }
        return valuesEqual;
    }

    /**
     * Применяет изменение к объекту если он еще не был изменен.
     * Если новое значение равно оригинальному, изменение игнорируется.
     *
     * @param changedProp      изменяемое свойство
     * @param newValueSupplier новое значение
     */
    public <V> void modifyIfNotChanged(ModelProperty<? super M, V> changedProp, Supplier<V> newValueSupplier) {
        if (!changed(changedProp)) {
            modify(changedProp, newValueSupplier.get());
        }
    }

    /**
     * Применяет изменение к объекту если выполняетя условие.
     * Если новое значение равно оригинальному, изменение игнорируется.
     *
     * @param changedProp изменяемое свойство
     * @param newValue    новое значение
     * @param predicate   условие
     */
    public <V> void modifyIf(ModelProperty<? super M, V> changedProp, @Nullable V newValue, Predicate<M> predicate) {
        if (predicate.test(model)) {
            modify(changedProp, newValue);
        }
    }

    /**
     * Применяет изменение к объекту если выполняетя условие и если объект пренадлежит к указанному
     * сабклассу/интерфейсу.
     * Если новое значение равно оригинальному, изменение игнорируется.
     *
     * @param changedProp изменяемое свойство
     * @param newValue    новое значение
     * @param predicate   условие
     * @param clazz       сабкласс
     */
    @SuppressWarnings("unchecked")
    public <V, T extends Model> void modifyIf(ModelProperty<? super T, V> changedProp, @Nullable V newValue,
                                              Class<T> clazz, Predicate<T> predicate) {
        if (clazz.isAssignableFrom(model.getClass())) {
            ((AppliedChanges) this).modifyIf(changedProp, newValue, predicate);
        }
    }

    /**
     * @param property свойство
     * @return оригинальное значение указанного свойства {@code property} у объекта модели
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public <V> V getOldValue(ModelProperty<? super M, V> property) {
        if (oldValues.containsKey(property)) {
            //noinspection SuspiciousMethodCalls
            return (V) oldValues.get(property);
        }
        return property.get(model);
    }

    /**
     * @param property свойство
     * @return значение указанного свойства {@code property} у объекта модели после применения изменений
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public <V> V getNewValue(ModelProperty<? super M, V> property) {
        return property.get(model);
    }

    /**
     * @return {@link Set}, содержащий в себе те свойства, которые были изменены предыдущими вызовами
     * {@link #modify(ModelProperty, Object)}
     */
    @Nonnull
    public Set<ModelProperty<? super M, ?>> getActuallyChangedProps() {
        return unmodifiableSet(oldValues.keySet());
    }

    /**
     * @return {@link Set}, содержащий в себе те свойства, которые нужно будет обновить
     * {@link #modify(ModelProperty, Object)}
     */
    @Nonnull
    public Set<ModelProperty<? super M, ?>> getPropertiesForUpdate() {
        Set<ModelProperty<? super M, ?>> result = new HashSet<>();
        result.addAll(oldValues.keySet());
        result.addAll(affectedSensitiveProperties);
        return unmodifiableSet(result);
    }

    /**
     * @return true, если имеются реально изменившиеся свойства
     */
    public boolean hasActuallyChangedProps() {
        return !oldValues.isEmpty();
    }

    /**
     * Версия метода {@link AppliedChanges#changed} для коллекций, которая отличается тем, что не различает null и
     * пустую коллекцию.
     * <p>Для коллекций {@code null} и {@code []} могут серилизоваться в одно и то же значение (обычно {@code null}
     * или отсутствие строки). В таком случае {@link AppliedChanges#changed} может возвращать {@code true} даже в тех
     * случаях, когда в базе ничего не поменяется и пользователь ничего не передавал (гриды передают полные объекты
     * независимо от того, что поменял пользователь).
     * <p>Для коллекций рекомендуется использовать этот метод вместо {@link AppliedChanges#changed}, за исключением
     * случаев, когда оба значения {@code null} и {@code []} действительно имеют смысл и различны.
     */
    public boolean collectionContentChanged(ModelProperty<? super M,
            ? extends Collection<?>> property) {
        //noinspection rawtypes
        return changed(property)
                && !CollectionUtils.isAllEmpty((Collection) oldValues.get(property), getNewValue(property));
    }

    /**
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было изменено
     */
    public boolean changed(ModelProperty<? super M, ?> property) {
        return oldValues.containsKey(property);
    }

    /**
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было изменено и новое значение – {@code null}.
     */
    public boolean deleted(ModelProperty<? super M, ?> property) {
        return changed(property) && getNewValue(property) == null;
    }

    /**
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было изменено с {@code null} на не {@code null}.
     */
    public boolean assigned(ModelProperty<? super M, ?> property) {
        return changed(property) && getOldValue(property) == null;
    }

    /**
     * Эквивалентно assigned || replaced
     *
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было изменено и новое значение – не {@code null}.
     */
    public boolean changedAndNotDeleted(ModelProperty<? super M, ?> property) {
        return changed(property) && getNewValue(property) != null;
    }

    /**
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было изменено и старое и новое значения – не
     * {@code null}.
     */
    public boolean replaced(ModelProperty<? super M, ?> property) {
        return changed(property) && getNewValue(property) != null && getOldValue(property) != null;
    }

    /**
     * Возвращает, было ли передано хотя бы один раз указанное свойство {@code property} в метод
     * {@link #modify(ModelProperty, Object)} либо один из его перегруженных вариантов.
     * В отличие от {@link #changed(ModelProperty)} возвращает {@code true}, если свойство передано, но совпадает
     * со значением в модели.
     * <p>Это полезно в валидации - если свойство передано, значит есть какой-то элемент, откуда оно передано
     * и есть смысл его валидировать и показывать ошибку (даже если пользователь его не менял и оно совпадает со
     * значением в базе).
     *
     * @param property {@link ModelProperty}
     * @return {@code true}, если указанное свойство {@code property} было передано
     */
    public boolean passed(ModelProperty<? super M, ?> property) {
        return passedProperties.contains(property);
    }

    /**
     * @return объект модели
     */
    @Nonnull
    public M getModel() {
        return model;
    }

    /**
     * Изменить тип модели
     * <p>
     * Полезно в случае работы с иерархиями моделей
     */
    @SuppressWarnings("unchecked")
    public <T extends Model> AppliedChanges<T> castModelUp(Class<T> clazz) {
        checkArgument(clazz.isAssignableFrom(model.getClass()));
        return (AppliedChanges) this;
    }

    @Override
    public String toString() {
        List<String> lines = new ArrayList<>();
        EntryStream.of(oldValues).forKeyValue((prop, oldValue) -> {
            lines.add(String.format("%s: %s -> %s", prop.name(), oldValue, getNewValue(prop)));
        });
        lines.sort(Comparator.naturalOrder());
        return "(" + String.join(", ", lines) + ")";
    }
}
