package ru.yandex.direct.model;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Для методов типа update нам нужно передавать требуемые изменения объектов
 * (у объекта 1 поменялись поля такие-то на такие-то значения)
 * <p>
 * ModelChanges представляет из себя набор изменений одного конкретного объекта
 * (его id хранится тут же).
 * <p>
 * При этом ModelChanges ничего не знает про фактический тип объекта, а работает c интерфейсом.
 *
 * @param <M>
 */
@ParametersAreNonnullByDefault
public class ModelChanges<M extends ModelWithId> {

    /**
     * Идентификатор объекта, изменения для которого храним
     */
    private final Long id;

    /**
     * Изменяемая модель. Может быть интерфейсом.
     */
    private final Class<? extends M> modelType;

    /**
     * Ассоциативный массив пропертей и их новых значений
     */
    private final Map<ModelProperty<? super M, ?>, Object> changedProps = new HashMap<>();

    /**
     * Инстанцировать изменения некоторой модели, запомнив её идентификатор и тип.
     *
     * @param id        id изменяемой модели
     * @param modelType тип модели, может быть интерфейсом
     */
    public ModelChanges(Long id, Class<M> modelType) {
        this.id = id;
        this.modelType = modelType;
    }

    /**
     * Инстанцировать изменения модели, запомнив изменение одного из её свойств.
     * Ссылка на объект модели не сохраняется, дальнейшие изменения объекта не приведут
     * к изменению состояния созданного инстанса {@link ModelChanges}.
     *
     * @param model    изменяемая модель
     * @param property изменяемое свойство
     * @param value    новое значение свойства модели
     * @param <M>      тип модели
     * @param <V>      тип свойства
     * @return Новый экземпляр {@link ModelChanges} с уже обработанным изменением проперти.
     * @see ModelProperty
     */
    @SuppressWarnings("unchecked")
    public static <M extends ModelWithId, V> ModelChanges<M> build(M model, ModelProperty<? super M, V> property,
                                                                   @Nullable V value) {
        //noinspection unchecked
        return new ModelChanges<>(model.getId(), (Class<M>) model.getClass()).process(value, property);
    }

    /**
     * Инстанцировать изменения некоторой модели, запомнив изменение одного из её свойств.
     *
     * @param id        id изменяемой модели
     * @param modelType класс, определяющий тип модели
     * @param property  изменяемое свойство
     * @param value     новое значение свойства модели
     * @param <M>       тип модели
     * @param <V>       тип свойства
     * @return Новый экземпляр {@link ModelChanges} с уже обработанным изменением проперти.
     * @see ModelProperty
     */
    public static <M extends ModelWithId, V> ModelChanges<M> build(
            long id, Class<M> modelType,
            ModelProperty<? super M, V> property, @Nullable V value
    ) {
        return new ModelChanges<>(id, modelType).process(value, property);
    }

    public static <X extends ModelWithId, V> Predicate<ModelChanges<X>> isPropertyChanged(
            ModelProperty<? super X, V> modelProperty) {
        return xModelChanges -> xModelChanges.isPropChanged(modelProperty);
    }

    public static <X extends ModelWithId, V> Function<ModelChanges<X>, V> getChangedProperty(
            ModelProperty<? super X, V> modelProperty) {
        return xModelChanges -> xModelChanges.getChangedProp(modelProperty);
    }

    public static <X extends ModelWithId, V> Consumer<ModelChanges<X>> propertyModifier(
            ModelProperty<? super X, V> property,
            UnaryOperator<V> modifier) {
        return xModelChanges -> xModelChanges.replace(property, modifier);
    }

    public static <X extends ModelWithId, V> Consumer<ModelChanges<X>> propertyModifier(
            ModelProperty<? super X, V> property,
            BiFunction<Long, V, V> modifier) {
        return xModelChanges -> xModelChanges.replace(property, modifier);
    }


    public <X extends M> AppliedChanges<X> applyTo(X object) {
        return applyTo(object, Collections.emptySet());
    }

    public <X extends M> AppliedChanges<X> applyTo(X object, Set<ModelProperty<? super X, ?>> sensitiveProperties) {
        return applyTo(object, sensitiveProperties, Collections.emptyMap());
    }

    public <X extends M> AppliedChanges<X> applyTo(
            X object,
            Set<ModelProperty<? super X, ?>> sensitiveProperties,
            Map<ModelProperty<? super X, ?>, BiFunction<Object, Object, Boolean>> customModelEquals
    ) {
        checkArgument(Objects.equals(id, object.getId()),
                "Can't apply modification to object with different ID. Expected ID: %s, but was: %s",
                id, object.getId());
        AppliedChanges<X> appliedChanges = new AppliedChanges<>(object, sensitiveProperties, customModelEquals);

        changedProps.forEach((changedProp, newValue) -> {
            @SuppressWarnings("unchecked")
            ModelProperty<M, Object> changedPropUnchecked = (ModelProperty<M, Object>) changedProp;
            appliedChanges.modify(changedPropUnchecked, newValue);
        });
        return appliedChanges;
    }

    /**
     * Возвращает класс изменяемой модели, может быть интерфейсом.
     */
    public Class<? extends M> getModelType() {
        return modelType;
    }

    /**
     * Записать изменение проперти property на значение val
     *
     * @param val      - новое значение проперти (возможно null)
     * @param property - пропертя
     * @param <V>
     * @return - ссылку на исходный объект ModelChanges (для fluent-интерфейса)
     */
    @Nonnull
    public <V> ModelChanges<M> process(@Nullable V val, ModelProperty<? super M, V> property) {
        changedProps.put(property, val);
        return this;
    }

    /**
     * Записать изменение проперти property на значение val
     *
     * @param val         - новое значение проперти (возможно null)
     * @param property    - пропертя
     * @param transformer - преобразование исходного значения в тип проперти
     * @param <V>
     * @return - ссылку на исходный объект ModelChanges (для fluent-интерфейса)
     */
    @Nonnull
    public <F, V> ModelChanges<M> process(@Nullable F val, ModelProperty<? super M, V> property,
                                          Function<F, V> transformer) {
        changedProps.put(property, transformer.apply(val));
        return this;
    }

    /**
     * Если val не null - записать изменение проперти property на значение val
     *
     * @param val      - новое значение проперти
     * @param property - пропертя
     * @param <V>
     * @return - ссылку на исходный объект ModelChanges (для fluent-интерфейса)
     */
    @Nonnull
    public <V> ModelChanges<M> processNotNull(@Nullable V val, ModelProperty<? super M, V> property) {
        if (val != null) {
            changedProps.put(property, val);
        }
        return this;
    }

    /**
     * Если val не null - применить к нему функцию и записать изменение проперти property
     *
     * @param val         - новое значение проперти
     * @param property    - пропертя
     * @param transformer - преобразование исходного значения в тип проперти
     * @param <V>
     * @return - ссылку на исходный объект ModelChanges (для fluent-интерфейса)
     */
    @Nonnull
    public <F, V> ModelChanges<M> processNotNull(@Nullable F val, ModelProperty<? super M, V> property,
                                                 Function<F, V> transformer) {
        if (val != null) {
            changedProps.put(property, transformer.apply(val));
        }
        return this;
    }

    /**
     * <ul>
     * <li>Если {@code val == null}, то не изменять значение {@code property}</li>
     * <li>Если {@code val == Optional.empty()}, то изменить значение {@code property} на {@code null}</li>
     * <li>Иначе изменить значение {@code property} на значение {@code val}</li>
     * <ul>
     * <p>
     * Пояснение: {@code Optional<V>} помогает в случае GraphQL запросов на мутацию различать ситуацию,
     * когда поле не передано, потому что его значение не нужно обновлять ({@code val == null})
     * (точечная мутация -- в запросе передается только то, что хотим обновить, а не весь объект целиком),
     * от ситуации, когда в поле явно передано значение {@code Optional.empty()}
     * с намерением затереть это поле у объекта.
     *
     * @param val      - новое значение проперти
     * @param property - пропертя
     * @return - ссылку на исходный объект ModelChanges (для fluent-интерфейса)
     */
    @Nonnull
    public <V> ModelChanges<M> processOptional(@Nullable Optional<V> val, ModelProperty<? super M, V> property) {
        //noinspection OptionalAssignedToNull
        if (val == null) {
            return this;
        }
        changedProps.put(property, val.orElse(null));
        return this;
    }

    /**
     * Возвращает поменялась ли указанная пропертя. Ничего не знает про текущее значение и возвращает true, даже если
     * новое значение совпадает с текущим.
     */
    public boolean isPropChanged(ModelProperty prop) {
        return changedProps.containsKey(prop);
    }

    public boolean isPropDeleted(ModelProperty prop) {
        return changedProps.containsKey(prop) && changedProps.get(prop) == null;
    }

    public boolean isPropChangedAndNotDeleted(ModelProperty prop) {
        return changedProps.get(prop) != null;
    }

    public <V> V replace(ModelProperty<? super M, V> property, UnaryOperator<V> modifier) {
        V oldValue = getChangedProp(property);
        V newValue = modifier.apply(oldValue);
        process(newValue, property);
        return newValue;
    }

    public <V> V replace(ModelProperty<? super M, V> property, BiFunction<Long, V, V> modifier) {
        V oldValue = getChangedProp(property);
        V newValue = modifier.apply(getId(), oldValue);
        process(newValue, property);
        return newValue;
    }

    /**
     * @return - поменялась ли хотя бы одна пропертя
     */
    public boolean isAnyPropChanged() {
        return !changedProps.isEmpty();
    }

    /**
     * @param prop
     * @param <V>
     * @return - новое значение указанной проперти
     * @throws IllegalArgumentException - если изменения не было
     */
    @Nullable
    public <V> V getChangedProp(ModelProperty<? super M, V> prop) {
        if (!changedProps.containsKey(prop)) {
            throw new IllegalArgumentException(String.format("property %s is not changed on object (%s)%d", prop.name(),
                    prop.getModelClass().getName(), id));
        }
        @SuppressWarnings("unchecked")
        V ret = (V) changedProps.get(prop);
        return ret;
    }

    /**
     * Получить значение, если было изменено (в том числе null) или вернуть null, если значение не задано
     *
     * @param prop
     * @param <V>
     * @return - новое значение или null
     */
    @Nullable
    public <V> V getPropIfChanged(ModelProperty<? super M, V> prop) {
        return isPropChanged(prop) ? getChangedProp(prop) : null;
    }

    /**
     * @return - id исходного объекта
     */
    public Long getId() {
        return id;
    }

    /**
     * Преобразуем модель в один из родительских классов.
     * Лучше не использовать, будет удалено в DIRECT-128611
     */
    @Deprecated
    @SuppressWarnings("unchecked")
    public <R extends ModelWithId> ModelChanges<R> castModelUp(Class<R> clazz) {
        checkArgument(clazz.isAssignableFrom(modelType));
        return (ModelChanges) this;
    }

    /**
     * Преобразуем модель в любой класс.
     * После каста может получится, что например ModelChanges<BannerWithCreative> имеет modelType например
     * BannerWithHref, что не очень, т.к. BannerWithHref не extends BannerWithCreative
     * В DIRECT-128611 будем разбираться.
     * То ли удалим этот метод.
     * То ли сделаем Class<?> modelType вместо Class<? extends M> modelType;
     */
    public <R extends ModelWithId> ModelChanges<R> castModel(Class<R> clazz) {
        return (ModelChanges) this;
    }

    @Deprecated // используй applyTo
    private M toModel(Class<? extends M> clazz) {
        try {
            M model = clazz.getDeclaredConstructor().newInstance();
            model.setId(id);
            changedProps.forEach((changedProp, newValue) -> {
                @SuppressWarnings("unchecked")
                ModelProperty<M, Object> changedPropUnchecked = (ModelProperty<M, Object>) changedProp;
                changedPropUnchecked.set(model, newValue);
            });
            return model;
        } catch (InstantiationException | IllegalAccessException |
                NoSuchMethodException | InvocationTargetException
                e
        ) {
            throw new ModelInstantiationException(e);
        }
    }

    @Deprecated // modelType может хранить не конечный класс, удалить в DIRECT-128611
    public M toModel() {
        return toModel(modelType);
    }

    /**
     * Возвращает список имен измененных свойств моделей
     */
    public Set<ModelProperty<? super M, ?>> getChangedPropsNames() {
        return changedProps.keySet();
    }
}
