package ru.yandex.direct.model;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Ссылка на определённое свойство определённой модели.
 * Имя свойства модели должно соответствовать имени свойства бина (Bean Property).
 * Свойства интернированы - их можно сравнивать по ссылке.
 *
 * @param <M> тип модели
 * @param <V> тип свойства
 */
@ParametersAreNonnullByDefault
public class ModelProperty<M extends Model, V> {
    private static final ConcurrentHashMap<InternKey, ModelProperty> INTERN_TABLE = new ConcurrentHashMap<>();

    private final Class<M> modelClass;

    private final String propertyName;

    private final Function<M, V> getter;
    private final BiConsumer<M, V> setter;

    private ModelProperty(Class<M> modelClass, String propertyName,
                          Function<M, V> getter, BiConsumer<M, V> setter) {
        assertBeanPropertyExists(modelClass, propertyName);

        this.modelClass = modelClass;
        this.propertyName = propertyName;
        this.getter = getter;
        this.setter = setter;
    }

    /**
     * Создать или вернуть уже созданный и закешированный
     * объект ModelProperty для свойства определенной модели
     *
     * @param modelClass   класс модели
     * @param propertyName текстовое название проперти бина
     * @param <M>          тип модели
     * @param <V>          тип свойства
     * @return ModelProperty
     */
    public static <M extends Model, V> ModelProperty<M, V> create(Class<M> modelClass, String propertyName,
                                                                  Function<M, V> getter, BiConsumer<M, V> setter) {
        InternKey key = new InternKey(modelClass, propertyName);
        @SuppressWarnings("unchecked")
        ModelProperty<M, V> cachedModelProp = INTERN_TABLE.computeIfAbsent(key, k ->
                new ModelProperty<>(modelClass, propertyName, getter, setter));
        return cachedModelProp;
    }

    /**
     * Создать объект ModelProperty для свойства только для чтения
     *
     * @see #create(Class, String, Function, BiConsumer)
     */
    public static <M extends Model, V> ModelProperty<M, V> createReadOnly(
            Class<M> modelClass,
            String propertyName,
            Function<M, V> getter) {
        return create(
                modelClass,
                propertyName,
                getter,
                (m, v) -> {
                    throw new UnsupportedOperationException();
                });
    }

    private static <M extends Model> void assertBeanPropertyExists(Class<M> modelClass, String propertyName) {
        try {
            modelClass.getMethod("get" + capitalize(propertyName));
        } catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(
                    String.format("Bean %s has no property with name \"%s\"", modelClass.getName(), propertyName));
        }
    }

    public Class<M> getModelClass() {
        return modelClass;
    }

    @Nonnull
    public String name() {
        return propertyName;
    }

    @Nullable
    public V get(M model) {
        return getter.apply(model);
    }

    @Nullable
    @SuppressWarnings({"unchecked", "rawtypes"})
    public Object getRaw(Model model) {
        return ((ModelProperty)this).get(model);
    }

    public void set(M model, @Nullable V value) {
        setter.accept(model, value);
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    public void copyRaw(Model src, Model dst) {
        var p = (ModelProperty)this;
        p.set(dst, p.get(src));
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ModelProperty)) {
            return false;
        }

        ModelProperty<?, ?> that = (ModelProperty<?, ?>) o;

        return modelClass == that.modelClass
                && Objects.equals(propertyName, that.propertyName);
    }

    @Override
    public int hashCode() {
        int result = modelClass.hashCode();
        result = 31 * result + propertyName.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "ModelProperty{" +
                "modelClass=" + modelClass +
                ", propertyName='" + propertyName + '\'' +
                '}';
    }

    private static class InternKey {
        private final Class clazz;
        private final String fieldName;

        InternKey(Class clazz, String fieldName) {
            this.clazz = clazz;
            this.fieldName = fieldName;
        }

        @Override
        public boolean equals(@Nullable Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof InternKey)) {
                return false;
            }
            InternKey internKey = (InternKey) o;
            return Objects.equals(clazz, internKey.clazz) &&
                    Objects.equals(fieldName, internKey.fieldName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(clazz, fieldName);
        }
    }
}

