package ru.yandex.direct.model;

import java.util.List;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Интерфейс связи между сущностями, служит для маркировки связи, и содержит минимально необходимые операции -
 * установка и получение внешнего ключа, получение классов сущностей родителя и ребенка
 * @param <ChildT>
 * @param <ParentKeyT>
 */
@ParametersAreNonnullByDefault
public interface Relationship<ParentT extends Entity<ParentKeyT>, ChildT extends Entity<?>, ParentKeyT> {
    /**
     * Получить значение внешнего ключа из ребенка в связи
     * @param child сущность-потомок
     * @return значение ключа
     */
    @Nullable
    ParentKeyT getParentId(ChildT child);

    /**
     * Установить значение внешнего ключа на ребенке в связи
     * @param child сущность-потомок
     * @param id значение ключа
     */
    void setParentId(ChildT child, ParentKeyT id);

    /**
     * Получить класс сущности родителя в связи
     */
    Class<ParentT> getParentEntityClass();

    /**
     * Получить класс сущности ребенка в связи
     */
    Class<ChildT> getChildEntityClass();

    /**
     * Позволяет вызвать потребителя родительского класса и идентификаторов parentIdsConsumer, захватив корректные
     * классы родителя и его идентификатора. Снаружи связи это сделать не получится из-за потери связи между
     * TParent и TParentKey
     * @param entities список сущностей-детей, у которых нужно получить идентификаторы родителей
     * @param parentIdsConsumer потребитель родительского класса и идентификаторов.
     */
    default void consumeNotNullParentIds(
            List<ChildT> entities, BiConsumer<Class<ParentT>, List<ParentKeyT>> parentIdsConsumer) {
        List<ParentKeyT> parentEntityIds = entities.stream()
                .map(this::getParentId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        parentIdsConsumer.accept(getParentEntityClass(), parentEntityIds);
    }
}
