package ru.yandex.direct.moderation.client;

import java.nio.charset.StandardCharsets;
import java.util.List;

import com.fasterxml.jackson.databind.JavaType;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.moderation.client.model.CheckUserDomainsRequest;
import ru.yandex.direct.moderation.client.model.CommonModerationRequest;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonRpcRequest;
import ru.yandex.direct.utils.JsonRpcResponse;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static io.netty.handler.codec.http.HttpHeaderValues.APPLICATION_JSON;
import static org.asynchttpclient.util.HttpConstants.Methods.POST;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.getTypeFactory;

/**
 * Клиент к json-rpc ручкам модерации
 */

public class ModerationClient {
    private static final Logger logger = LoggerFactory.getLogger(ModerationClient.class);
    private static final int DEFAULT_REQUEST_ID = 1;

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final ModerationClientConfiguration config;

    public ModerationClient(ModerationClientConfiguration config, AsyncHttpClient asyncHttpClient) {
        this.config = config;
        FetcherSettings settings = new FetcherSettings()
                .withRequestTimeout(config.getReadTimeout());
        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings);

    }

    public ModerationClient(ModerationClientConfiguration config, ParallelFetcherFactory parallelFetcherFactory) {
        this.config = config;
        this.parallelFetcherFactory = parallelFetcherFactory;
    }

    /**
     * Сделать запрос к json-rpc ручке {@code requestMethod} Модерации и вернуть ее ответ в виде объекта {@code resultClass}
     *
     * @throws ModerationClientException если RPC был неуспешен или при ошибке десериализации
     */
    <T> T get(String requestMethod, CommonModerationRequest params, Class<T> resultClass) {
        JsonRpcRequest<CommonModerationRequest> request = new JsonRpcRequest<>(requestMethod, params,
                DEFAULT_REQUEST_ID);
        String jsonRpcResponse = doRequest(request);

        try {
            JavaType responseType = getTypeFactory().constructParametricType(JsonRpcResponse.class, resultClass);
            JsonRpcResponse<T> response = fromJson(jsonRpcResponse, responseType);
            if (response.getError() != null) {
                logger.info("Got errors on request {}.", request);
                throw new ModerationClientException("Got error on response for JSON-RPC request",
                        new Throwable(response.getError().getMessage()));
            } else if (response.getResult() == null) {
                logger.info("Got \"null\" as a result on request {}.", request);
                throw new ModerationClientException("Got \"null\" as a result for JSON-RPC request");
            }
            return response.getResult();
        } catch (IllegalArgumentException ex) {
            throw new ModerationClientException("Got error while parsing response", ex);
        }
    }

    /**
     * Сделать запрос к json-rpc ручке Модерации по входным параметрам и вернуть ее ответ в строки (json)
     */
    private String doRequest(JsonRpcRequest<CommonModerationRequest> request) {
        Result<String> jsonRpcResult;
        try (ParallelFetcher<String> fetcher = parallelFetcherFactory.getParallelFetcher();
             TraceProfile profile = Trace.current().profile("moderation_client:" + request.getMethod())) {
            RequestBuilder builder = new RequestBuilder(POST)
                    .setUrl(config.getJsonRpcUrl())
                    .setCharset(StandardCharsets.UTF_8)
                    .setBody(request.toString())
                    .addHeader(CONTENT_TYPE, APPLICATION_JSON);
            ParsableRequest<String> asyncRequest = new ParsableStringRequest(request.getId(), builder.build());

            jsonRpcResult = fetcher.execute(asyncRequest);
            if (jsonRpcResult.getSuccess() == null || (jsonRpcResult.getErrors() != null && !jsonRpcResult.getErrors()
                    .isEmpty())) {
                logger.info("Got errors on request {}.", request);
                RuntimeException ex = new ModerationClientException("Got error on response for JSON-RPC request");
                if (jsonRpcResult.getErrors() != null && !jsonRpcResult.getErrors().isEmpty()) {
                    jsonRpcResult.getErrors().forEach(ex::addSuppressed);
                }
                throw ex;
            }
        } catch (Exception ex) {
            logger.info("Request {} were unexpectedly interrupted.", request);
            Thread.currentThread().interrupt();
            throw new ModerationClientException(ex.getMessage(), ex);
        }

        return jsonRpcResult.getSuccess();
    }

    /**
     * Проверяет домены и/или номера телефонов в списке предупреждений
     */
    public boolean hasDomainsOrPhonesNotInWhitelist(
            List<String> domains, List<String> phones, long uid, long clientId) {
        CheckUserDomainsRequest req = new CheckUserDomainsRequest(domains, phones, uid, clientId);
        return get("checkUserDomains", req, Integer.class) == 0;
    }
}
