package ru.yandex.direct.moderation.client;

import java.time.Duration;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;

/**
 * Конфигурация для jsonrpc-клиента к модерации
 * <p>
 * Как использовать:
 * <pre> {@code
 * ModerationClientConfiguration config = new ModerationClientConfiguration("http://direct-mod-test.yandex-team.ru:7080/jsonrpc", Duration.ofSeconds(180));
 * }
 * или без timeout-а:
 * {@code
 * ModerationClientConfiguration config = new ModerationClientConfiguration("http://direct-mod-test.yandex-team.ru:7080/jsonrpc");
 * }
 * </pre>
 */

public class ModerationClientConfiguration {
    public static final Duration DEFAULT_READ_TIMEOUT = Duration.ofSeconds(180);

    private String jsonRpcUrl;
    private Duration readTimeout;

    public ModerationClientConfiguration(String jsonRpcUrl) {
        this(jsonRpcUrl, DEFAULT_READ_TIMEOUT);
    }

    public ModerationClientConfiguration(String jsonRpcUrl, Duration readTimeout) {
        checkArgument(!isNullOrEmpty(jsonRpcUrl));
        checkNotNull(readTimeout);
        checkArgument(!(readTimeout.isNegative() || readTimeout.isZero()));

        this.jsonRpcUrl = jsonRpcUrl;
        this.readTimeout = readTimeout;
    }

    public String getJsonRpcUrl() {
        return this.jsonRpcUrl;
    }

    public Duration getReadTimeout() {
        return this.readTimeout;
    }
}
