package ru.yandex.direct.multitype.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.multitype.repository.container.AddOrUpdateAndDeleteContainer;

/**
 * Абстрактный класс для реализации на базе него репозиторных тайп-саппортов конкретных ассетов.
 * <p>
 * Подходит для отношения один-ко-многим к основной таблице.
 * <p>
 * Для вставки и обновления должна использоваться конструкция insert ... on duplicate key update,
 * реализуемая в наследнике.
 * <p>
 * Поддерживает хранение полезных данных в двух видах:
 * - в полях контейнера, который хранится в интерфейсе модели в виде списка (пример: {@code List<Container>})
 * - непосредственно в списке, когда в смежной таблице всего одно поле (пример {@code List<String>}
 * <p>
 * ВАЖНО: используйте хранение значения непосредственно в списке без промежуточного контейнера только если есть
 * высокая степень уверенности, что в смежную таблицу и в модель не будут добавляться новые поля.
 *
 * @param <T> - интерфейс
 * @param <E> - модель для представления одной строки в базе, для использования в jooqMapper
 */
@ParametersAreNonnullByDefault
public abstract class AbstractMultiRowEntityRepositoryTypeSupport
        <T extends Model, A, U, E extends Model>
        extends AbstractRepositoryTypeSupport<T, A, U> {

    protected AbstractMultiRowEntityRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public final void updateAdditionTables(DSLContext context, U updateParameters,
                                           Collection<AppliedChanges<T>> modelsChangesCollection) {
        AddOrUpdateAndDeleteContainer<E> container = buildAddOrUpdateAndDeleteContainer(context,
                updateParameters, modelsChangesCollection);
        // важно сначала удалить, потом добавлять, т.к. в rowsToDelete и rowsToAddOrUpdate могут быть записи с
        // одинаковым primary key, а имплементация при удалении может использовать только primary key,
        // не проверяя остальные поля
        if (container.getRowsToDelete().size() > 0) {
            delete(context, container.getRowsToDelete());
        }
        if (container.getRowsToAddOrUpdate().size() > 0) {
            addOrUpdate(context, container.getRowsToAddOrUpdate());
        }
    }

    /**
     * Метод для построения контейнера, который предоставляет информацию о том, какие записи необходимо удалить,
     * а какие добавить или обновить с помощью insert ... on duplicate key update.
     * <p>
     * За более детальной информацией обращайтесь к javadoc к реализациям AddOrUpdateAndDeleteContainer
     * и уже реализованным наследникам.
     */
    protected abstract AddOrUpdateAndDeleteContainer<E> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            U updateParameters,
            Collection<AppliedChanges<T>> modelsChangesCollection);

    @Override
    public final void insertToAdditionTables(DSLContext context, A addModelParametersContainer,
                                             Collection<T> models) {
        List<E> newRepositoryModels = extractNewRepositoryModels(addModelParametersContainer, models);

        if (newRepositoryModels.size() > 0) {
            addOrUpdate(context, newRepositoryModels);
        }
    }

    /**
     * Преобразовать полезные данные интерфейса из списка моделей
     * в список объектов-представлений для работы с jooqMapper-ом.
     */
    @Nonnull
    protected abstract List<E> extractNewRepositoryModels(A addModelParametersContainer, Collection<T> models);

    /**
     * Обработчик на удаление. Будет вызван только если коллекция не пустая.
     * <p>
     * Записи из repositoryModelsToDelete гарантированно есть в таблице именно в таком виде, поэтому при удалении
     * можно смотреть только на primary key и игнорировать остальные поля.
     */
    protected abstract void delete(DSLContext context, Collection<E> repositoryModelsToDelete);

    /**
     * Обработчик на добавление или обновление. Будет вызван только если коллекция не пустая.
     */
    protected abstract void addOrUpdate(DSLContext context, Collection<E> repositoryModelsToAddOrUpdate);

    @Override
    public abstract void enrichModelFromOtherTables(DSLContext dslContext, Collection<T> models);

    @Override
    public final Collection<Field<?>> getFields() {
        return Collections.emptyList();
    }

    @Override
    public final <M extends T> void fillFromRecord(M model, Record record) {
    }
}
