package ru.yandex.direct.multitype.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.multitype.entity.JoinQuery;
import ru.yandex.direct.multitype.typesupport.TypeSupport;

/**
 * @param <T> - конечный тип, который обрабатывается репозиторием
 * @param <A> - контейнер, который передается в методы добавления
 * @param <U> - контейнер, который передается в методы обновления
 */
@ParametersAreNonnullByDefault
public interface RepositoryTypeSupport<T extends Model, A, U> extends TypeSupport<T> {
    /**
     * Возвращает поля, которые будут запрашиваться из базы при SELECT Model из базы.
     */
    Collection<Field<?>> getFields();

    /**
     * Метод возвращающий джойны необходимый для извлечения модели.
     * По дефолту пустой список.
     * Для необязательных таблиц в сабкласах следует использовать исключительно LEFT_OUTER_JOIN, иначе массовые
     * selectы не будут работать.
     * Для обязательных таблиц в сабкласах кажется логичным использовать INNER_JOIN.
     */
    default List<JoinQuery> joinQuery() {
        return Collections.emptyList();
    }

    /**
     * Заполняем модель реализующую интерфейс T record'ом.
     */
    <M extends T> void fillFromRecord(M model, Record record);

    /**
     * Дополняем модель данными из таблиц не входящих в addSelectJoin, например: ppc.metrika_counters
     */
    void enrichModelFromOtherTables(DSLContext dslContext, Collection<T> models);

    /**
     * Делаем запрос в базу на обновление модели в тех таблицах, которые могут отсутствовать модели
     */
    void updateAdditionTables(DSLContext context, U updateParameters,
                              Collection<AppliedChanges<T>> appliedChanges);

    /**
     * Делаем запрос в базу на добавление моделей в тех таблицах, которые могут отсутствовать модели
     */
    void insertToAdditionTables(DSLContext context, A addModelParametersContainer,
                                Collection<T> models);

    /**
     * Добавляем новые поля в InsertHelper'ы по таблицам
     */
    void pushToInsert(InsertHelperAggregator insertHelperAggregator, T model);

    /**
     * Добавляем новые поля в Update'ы по таблицам
     */
    void processUpdate(UpdateHelperAggregator updateHelperAggregator, Collection<AppliedChanges<T>> appliedChanges);

}
