package ru.yandex.direct.multitype.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectQuery;

import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.multitype.entity.JoinCollector;
import ru.yandex.direct.multitype.typesupport.TypeSupport;
import ru.yandex.direct.multitype.typesupport.TypeSupportAffectionHelper;
import ru.yandex.direct.multitype.typesupport.TypeSupportUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.multitype.typesupport.TypeSupportUtils.getObjectsByTypeSupports;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public abstract class RepositoryTypeSupportFacade<T extends Model, K, A, U> {
    private final Set<Field<?>> allModelFields;
    private final List<? extends RepositoryTypeSupport<? extends T, A, U>> supports;
    private final Map<K, Supplier<T>> initializerByType;
    private final Collection<Field<?>> additionalReadFields;
    private final TypeSupportAffectionHelper<T> typeSupportAffectionHelper;


    public RepositoryTypeSupportFacade(List<? extends RepositoryTypeSupport<? extends T, A, U>> supports,
                                       Map<K, Supplier<T>> initializerByType,
                                       Collection<Field<?>> additionalReadFields,
                                       Set<Class<? extends T>> whiteListSupportTypeClass) {
        this.supports = supports;
        this.initializerByType = initializerByType;
        this.additionalReadFields = additionalReadFields;

        this.allModelFields = new HashSet<>(additionalReadFields);
        this.allModelFields.addAll(collectAllModelFields());
        this.typeSupportAffectionHelper = new TypeSupportAffectionHelper<>(whiteListSupportTypeClass);
    }

    private Collection<Field<?>> collectAllModelFields() {
        return StreamEx.of(supports)
                .map(RepositoryTypeSupport::getFields)
                .flatMap(StreamEx::of)
                .toSet();
    }

    /**
     * Получить все поля из базы, относящиеся к модели
     */
    public Collection<Field<?>> getAllModelFields() {
        return allModelFields;
    }

    /**
     * Получить поля из базы, относящиеся к указанным моделям
     */
    public Collection<Field<?>> getModelFields(Collection<Class<? extends T>> classes) {
        List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                TypeSupportUtils.getSupportsByClasses(supports, classes);
        return StreamEx.of(relevantSupports)
                .map(RepositoryTypeSupport::getFields)
                .append(additionalReadFields)
                .flatMap(Collection::stream)
                .toSet();
    }

    /**
     * Получить все поддерживаемые типы
     */
    public Set<K> getSupportedTypes() {
        return Collections.unmodifiableSet(initializerByType.keySet());
    }

    @Nullable
    protected Condition getConditionThatRecordAnyOfClasses(Collection<Class<? extends T>> classes) {
        return null;
    }

    /**
     * Получить экземпляр модели по записи в базе
     */
    public T getModelFromRecord(Record record) {
        T model = getModelInitializer(getModelType(record)).get();
        List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports = getSupportsByClass(model);
        fillModelFromRecord(record, model, relevantSupports);
        return model;
    }

    /**
     * Получить экземпляр модели по записи в базе
     * Заполняются только поля необходимые в указанном классе
     */
    public <M extends T> M getModelFromRecord(Record record, Class<M> clazz) {
        return (M) getModelFromRecord(record, Collections.singletonList(clazz));
    }

    /**
     * Получить экземпляр модели по записи в базе
     * Заполняются только поля необходимые в указанных классах
     */
    public T getModelFromRecord(Record record, Collection<Class<? extends T>> classes) {
        T model = getModelInitializer(getModelType(record)).get();
        Class<? extends Model> modelClass = model.getClass();
        classes.forEach(clazz -> checkArgument(clazz.isAssignableFrom(modelClass),
                "Model %s doesn't implement/extend requested %s", modelClass, clazz));

        List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                TypeSupportUtils.getSupportsByClasses(supports, classes);
        fillModelFromRecord(record, model, relevantSupports);
        return model;
    }

    private void fillModelFromRecord(Record record,
                                     T model,
                                     List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        relevantSupports.forEach(support -> {
            if (isRecordFilled(support.getFields(), record)) {
                ((RepositoryTypeSupport) support).fillFromRecord(model, record);
            }
        });
    }

    protected abstract K getModelType(Record record);

    private Supplier<T> getModelInitializer(K type) {
        checkArgument(initializerByType.containsKey(type), "Unsupported type: " + type.toString());
        return initializerByType.get(type);
    }


    private List<? extends RepositoryTypeSupport<? extends T, A, U>> getSupportsByClass(Model model) {
        return TypeSupportUtils.getSupportsByClass(supports, model.getClass());
    }

    /**
     * Проверяем, есть ли в рекорде заполненные поля
     */
    private static boolean isRecordFilled(Collection<Field<?>> fields, Record record) {
        return StreamEx.of(fields)
                .map(record::getValue)
                .nonNull()
                .findAny()
                .isPresent();
    }

    /**
     * Дозаполнить модели.
     * Необходимо в ситуации когда связанная таблиц относится к базовой таблице, как многие ко многим
     */
    public void enrichModelFromOtherTables(DSLContext dslContext, List<T> models) {
        enrichModelFromOtherTables(dslContext, models, supports);
    }

    /**
     * Дозаполнить модели.
     * Необходимо в ситуации когда связанная таблиц относится к базовой таблице, как многие ко многим
     * Заполняются только поля необходимые в указанном классе
     */
    public void enrichModelFromOtherTables(DSLContext dslContext, List<T> models, Class<? extends T> clazz) {
        enrichModelFromOtherTables(dslContext, models, Collections.singletonList(clazz));
    }

    /**
     * Дозаполнить модели.
     * Необходимо в ситуации когда связанная таблиц относится к базовой таблице, как многие ко многим
     * Заполняются только поля необходимые в указанных классах
     */
    public void enrichModelFromOtherTables(DSLContext dslContext, List<T> models,
                                           Collection<Class<? extends T>> classes) {
        List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                TypeSupportUtils.getSupportsByClasses(supports, classes);
        enrichModelFromOtherTables(dslContext, models, relevantSupports);
    }

    private void enrichModelFromOtherTables(DSLContext dslContext,
                                            List<T> models,
                                            List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        getObjectsByTypeSupports(relevantSupports, models, T::getClass)
                .forEach((support, list) -> enrichModelFromOtherTables(support, dslContext, list));
    }

    private <M extends T> void enrichModelFromOtherTables(RepositoryTypeSupport<M, A, U> support,
                                                          DSLContext dslContext,
                                                          List<T> models) {
        List<M> typedModels = mapList(models, support.getTypeClass()::cast);
        if (!typedModels.isEmpty()) {
            support.enrichModelFromOtherTables(dslContext, typedModels);
        }
    }

    /**
     * Добавить в селект запрос необходимые join'ы
     */
    public <R extends Record> SelectQuery<R> collectSelectJoinStep(SelectQuery<R> query) {
        return collectSelectJoinStep(query, supports);
    }

    /**
     * Добавить в селект запрос необходимые join'ы
     * Добавляются только join'ы необходимые для указанного класса
     */
    public <R extends Record> SelectQuery<R> collectSelectJoinStep(SelectQuery<R> query, Class<? extends T> clazz) {
        return collectSelectJoinStep(query, Collections.singletonList(clazz));
    }

    /**
     * Добавить в селект запрос необходимые join'ы
     * Добавляются только join'ы необходимые для указанных классов
     */
    public <R extends Record> SelectQuery<R> collectSelectJoinStep(SelectQuery<R> query,
                                                                   Collection<Class<? extends T>> classes) {
        List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports =
                TypeSupportUtils.getSupportsByClasses(supports, classes);
        return collectSelectJoinStep(query, relevantSupports);
    }

    private <R extends Record> SelectQuery<R> collectSelectJoinStep(
            SelectQuery<R> query, List<? extends RepositoryTypeSupport<? extends T, A, U>> relevantSupports) {
        var collector = new JoinCollector();
        StreamEx.of(relevantSupports)
                .flatMap(support -> support.joinQuery().stream())
                .forEach(collector::collect);
        return collector.addSelectJoins(query);
    }

    public void insertToAdditionTables(DSLContext context,
                                       A addModelContainer,
                                       Collection<? extends T> models) {
        getModelsGroupedByTypeSupports(models)
                .forEach(((typeSupport, typedModels) ->
                        insertToAdditionTables(typeSupport, context, addModelContainer, typedModels)));
    }

    /**
     * Получить список изменений моделей по соответвующим Support'ам
     */
    private Map<? extends RepositoryTypeSupport<? extends T, A, U>,
            ? extends List<? extends T>> getModelsGroupedByTypeSupports(
            Collection<? extends T> models) {
        return getObjectsByTypeSupports(supports, models, Object::getClass);
    }

    private <M extends T> void insertToAdditionTables(
            RepositoryTypeSupport<M, A, U> support,
            DSLContext context,
            A addModelContainer,
            List<? extends T> models) {
        support.insertToAdditionTables(context, addModelContainer, mapList(models, c -> (M) c));
    }

    public void updateAdditionTables(DSLContext context,
                                     U updateContainer,
                                     Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges)
                .forEach(((typeSupport, changes) ->
                        updateAdditionTablesBySupport(typeSupport, context, updateContainer, changes)));

    }

    private Map<? extends RepositoryTypeSupport<? extends T, A, U>,
            ? extends List<? extends AppliedChanges<? extends T>>> getAppliedChangesGroupedByTypeSupports(
            Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        return StreamEx.of(supports)
                .mapToEntry(TypeSupport::getTypeClass)
                .mapValues(supportClass -> typeSupportAffectionHelper.selectAppliedChangesThatAffectSupport(
                        appliedChanges, supportClass))
                .removeValues(List::isEmpty)
                .toMap();
    }

    private <M extends T> void updateAdditionTablesBySupport(
            RepositoryTypeSupport<M, A, U> support,
            DSLContext context,
            U updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<AppliedChanges<M>> appliedChangesOfCertainModelType = mapList(appliedChanges,
                changes -> changes.castModelUp(support.getTypeClass()));
        if (!appliedChangesOfCertainModelType.isEmpty()) {
            support.updateAdditionTables(context, updateContainer, appliedChangesOfCertainModelType);
        }
    }

    public void processUpdate(UpdateHelperAggregator updateBuilderAggregator,
                              Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges)
                .forEach((support, changes) -> processUpdateBySupport(support, updateBuilderAggregator, changes));
    }

    private <M extends T> void processUpdateBySupport(
            RepositoryTypeSupport<M, A, U> support,
            UpdateHelperAggregator updateBuilderAggregator,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<AppliedChanges<M>> appliedChangesOfCertainModelType = mapList(appliedChanges,
                changes -> changes.castModelUp(support.getTypeClass()));
        support.processUpdate(updateBuilderAggregator, appliedChangesOfCertainModelType);
    }

    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, List<? extends T> modelList) {
        modelList.forEach(model -> pushToInsert(insertHelperAggregator, model));
    }

    private void pushToInsert(InsertHelperAggregator insertHelperAggregator,
                              Model model) {
        getSupportsByClass(model)
                .forEach(support -> pushToInsertBySupport(support, insertHelperAggregator, model));
        insertHelperAggregator.newRecord();
    }

    private <M extends T> void pushToInsertBySupport(
            RepositoryTypeSupport<M, A, U> support,
            InsertHelperAggregator insertHelperAggregator,
            Model model) {
        //noinspection unchecked
        support.pushToInsert(insertHelperAggregator, (M) model);
    }

    protected <M extends T> Predicate<Record> filterRecordIsSupportedByClass(Class<M> clazz) {
        return record -> clazz.isAssignableFrom(getModelInitializer(getModelType(record)).get().getClass());
    }

    protected Predicate<Class<? extends T>> filterClassIsAssignableFromRecord(Record record) {
        return clazz -> clazz.isAssignableFrom(getModelInitializer(getModelType(record)).get().getClass());
    }
}
