package ru.yandex.direct.multitype.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.SelectQuery;
import org.jooq.Table;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class TypeModifyRepository<S extends ModelWithId, T extends S,
        A extends RepositoryContainer, U extends RepositoryContainer> {

    private final DslContextProvider ppcDslContextProvider;
    private final RepositoryTypeSupportFacade<S, ?, A, U> typeSupportFacade;
    private final TypedRepository<S, ?, A, U> typedRepository;

    public TypeModifyRepository(DslContextProvider ppcDslContextProvider,
                                RepositoryTypeSupportFacade<S, ?, A, U> typeSupportFacade,
                                TypedRepository<S, ?, A, U> typedRepository) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.typeSupportFacade = typeSupportFacade;
        this.typedRepository = typedRepository;
    }

    public List<Long> add(DSLContext context,
                          A addContainer,
                          List<? extends T> models) {
        generateIds(addContainer, models);
        return addInternal(context, addContainer, models);
    }

    protected List<Long> addInternal(DSLContext context, A addContainer, List<? extends T> models) {
        context.transaction(configuration -> {
            DSLContext dsl = configuration.dsl();
            InsertHelperAggregator insertHelperAggregator = new InsertHelperAggregator(dsl);
            typeSupportFacade.pushToInsert(insertHelperAggregator, models);
            insertHelperAggregator.executeIfRecordsAdded();
            typeSupportFacade.insertToAdditionTables(dsl, addContainer, models);
        });
        return mapList(models, ModelWithId::getId);
    }

    protected abstract void generateIds(A addContainer, List<? extends T> models);

    public void update(U updateContainer,
                       Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        update(ppcDslContextProvider.ppc(updateContainer.getShard()), updateContainer, appliedChanges);
    }

    public void update(DSLContext dsl,
                       U updateParameters,
                       Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return;
        }

        updateTable(dsl, appliedChanges);
        typeSupportFacade.updateAdditionTables(dsl, updateParameters, appliedChanges);
    }

    protected void updateTable(DSLContext dsl,
                               Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<Long> modelIds = mapList(appliedChanges, changes -> changes.getModel().getId());
        lockModels(modelIds, dsl);

        UpdateHelperAggregator updateHelperAggregator = new UpdateHelperAggregator(dsl);
        typeSupportFacade.processUpdate(updateHelperAggregator, appliedChanges);
        updateHelperAggregator.execute();
    }

    /**
     * Обновляет только те модели, которые удовлетворяют переданному предикату после взятия лока и их повторного чтения
     * из базы. Обновление также затрагивает связанные таблицы
     */
    public <M extends T> int updateByPredicate(DSLContext dsl,
                                               U updateParameters,
                                               Collection<ModelChanges<M>> modelChanges,
                                               Class<M> modelClass,
                                               Predicate<? super M> predicate) {
        if (modelChanges.isEmpty()) {
            return 0;
        }

        Map<Long, ModelChanges<M>> idToModelChanges = listToMap(modelChanges, ModelChanges::getId);
        Collection<Long> modelIds = idToModelChanges.keySet();
        lockModels(modelIds, dsl);

        List<AppliedChanges<M>> appliedChanges =
                StreamEx.of(typedRepository.getStrictlyFullyFilled(dsl, modelIds, modelClass))
                        .filter(predicate)
                        .map(model -> idToModelChanges.get(model.getId()).applyTo(model))
                        .toList();

        if (appliedChanges.isEmpty()) {
            return 0;
        }

        UpdateHelperAggregator updateHelperAggregator = new UpdateHelperAggregator(dsl);
        typeSupportFacade.processUpdate(updateHelperAggregator, appliedChanges);
        updateHelperAggregator.execute();
        typeSupportFacade.updateAdditionTables(dsl, updateParameters, appliedChanges);

        return appliedChanges.size();
    }

    private void lockModels(Collection<Long> modelIds, DSLContext dsl) {
        SelectQuery<Record1<Integer>> query = dsl
                .selectZero()
                .from(getLockTable())
                .getQuery();

        typeSupportFacade.collectSelectJoinStep(query);

        var filter = getIdFilter(modelIds);
        filter.apply(query);

        query.setForUpdate(true);

        query.execute();
    }

    protected abstract Table<?> getLockTable();

    protected abstract Filter getIdFilter(Collection<Long> modelIds);
}
