package ru.yandex.direct.multitype.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectQuery;
import org.jooq.Table;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class TypedRepository<T extends ModelWithId, K,
        A extends RepositoryContainer, U extends RepositoryContainer> {

    protected final DslContextProvider dslContextProvider;

    private final RepositoryTypeSupportFacade<T, K, A, U> typeSupportFacade;

    private final Collection<Field<?>> allFields;

    public TypedRepository(DslContextProvider dslContextProvider,
                           RepositoryTypeSupportFacade<T, K, A, U> typeSupportFacade) {
        this.typeSupportFacade = typeSupportFacade;
        this.dslContextProvider = dslContextProvider;

        this.allFields = typeSupportFacade.getAllModelFields();

    }

    protected abstract Filter getIdFilter(Collection<Long> modelIds);

    protected abstract Table<?> getBaseTable();

    /**
     * Получить все поддерживаемые типы
     */
    public Set<K> getSupportedTypes() {
        return typeSupportFacade.getSupportedTypes();
    }

    public Map<Long, T> getIdToModelTyped(int shard, Collection<Long> modelIds) {
        return getIdToModelTyped(dslContextProvider.ppc(shard), modelIds);
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    public Map<Long, T> getIdToModelTyped(DSLContext dslContext, Collection<Long> modelIds) {
        var models = getTyped(dslContext, modelIds);
        return listToMap(models, ModelWithId::getId);
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    public List<T> getTyped(int shard, Collection<Long> modelIds) {
        return getTyped(dslContextProvider.ppc(shard), modelIds);
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    public List<T> getTyped(DSLContext dslContext, Collection<Long> modelIds) {
        return getTyped(dslContext, getIdFilter(modelIds));
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    protected List<T> getTypedByFilter(int shard, Filter filter) {
        return getTyped(dslContextProvider.ppc(shard), filter);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> Map<Long, M> getIdToModelSafely(DSLContext dslContext,
                                                         Collection<Long> modelIds,
                                                         Class<M> clazz) {
        List<M> models = getSafely(dslContext, modelIds, clazz);
        return listToMap(models, ModelWithId::getId);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> Map<Long, M> getIdToModelSafely(int shard,
                                                         Collection<Long> modelIds,
                                                         Class<M> clazz) {
        List<M> models = getSafely(shard, modelIds, clazz);
        return listToMap(models, ModelWithId::getId);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> List<M> getSafely(
            int shard,
            Collection<Long> modelIds,
            Class<M> clazz) {
        return getSafely(dslContextProvider.ppc(shard), modelIds, clazz);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> List<M> getSafely(
            DSLContext dslContext,
            Collection<Long> modelIds,
            Class<M> clazz) {
        return getSafely(dslContext, getIdFilter(modelIds), clazz);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> List<M> getSafely(int shard,
                                           Filter filter,
                                           Class<M> clazz) {
        return getSafely(dslContextProvider.ppc(shard), filter, clazz);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> List<M> getSafely(
            DSLContext dslContext,
            Filter filter,
            Class<M> clazz) {

        return (List<M>) getSafely(dslContext, filter, singletonList(clazz));
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public <M extends T> List<M> getSafely(
            DSLContext dslContext,
            Filter filter,
            LimitOffset limitOffset,
            Class<M> clazz) {

        return (List<M>) getSafely(dslContext, filter, limitOffset, singletonList(clazz));
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public List<T> getSafely(
            int shard,
            Collection<Long> modelIds,
            Collection<Class<? extends T>> classes) {
        return getSafely(shard, getIdFilter(modelIds), classes);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public List<T> getSafely(int shard,
                             Filter filter,
                             Collection<Class<? extends T>> classes) {
        return getSafely(dslContextProvider.ppc(shard), filter, classes);
    }

    /**
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     */
    public List<T> getSafely(
            DSLContext dslContext,
            Filter filter,
            Collection<Class<? extends T>> classes) {
        return getSafely(dslContext, filter, null, classes);
    }

    /**
     * Возвращает объекты удовлетворяющие filter И имплементирующие хотя бы один из указанных интерфейсов. При этом
     * заполняются поля всех (и только) указанных интерфейсов.
     * <p>
     * Например есть интерфейсы A, B, C, D и класс ABCImpl implements A,B,C. Запросили classes = (A, B, D) - вернётся
     * объект ABCImpl, в котором будут заполнены поля интерфейсов A и B, но не будет заполнен C.
     * Если объект удовлетворяет фильтру, но не имплементирует ни один из интерфейсов - он не будет возвращён (и
     * метод не выкинет exception, поэтому "safely").
     * <p>
     * За подробным описанием обращайтесь к странице
     * https://docs.yandex-team.ru/direct-dev/dev/banner/concept#repo-read
     *
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     * @see #getStrictly(DSLContext, Collection, Class)
     * @see #getStrictlyFullyFilled(DSLContext, Collection, Class)
     */
    public List<T> getSafely(
            DSLContext dslContext,
            Filter filter,
            @Nullable LimitOffset limitOffset,
            Collection<Class<? extends T>> classes) {
        if (filter.isEmpty()) {
            return emptyList();
        }

        SelectQuery<Record> query = selectClassFields(dslContext, classes);
        typeSupportFacade.collectSelectJoinStep(query, classes);

        filter.apply(query);
        var typeCondition = typeSupportFacade.getConditionThatRecordAnyOfClasses(classes);
        if (typeCondition != null) {
            query.addConditions(typeCondition);
        }
        if (limitOffset != null) {
            query.addOffset(limitOffset.offset());
            query.addLimit(limitOffset.limit());
        }

        Result<Record> records = query.fetch();
        List<T> models =
                StreamEx.of(records)
                        .mapToEntry(Function.identity(), r -> filterList(classes,
                                typeSupportFacade.filterClassIsAssignableFromRecord(r)))
                        .removeValues(List::isEmpty)
                        .mapKeyValue(typeSupportFacade::getModelFromRecord)
                        .toList();
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models, classes);

        return models;
    }

    /**
     * @see #getStrictly(DSLContext, Collection, Class)
     */
    public <M extends T> List<M> getStrictly(
            int shard,
            Collection<Long> modelIds,
            Class<M> clazz) {
        return getStrictly(dslContextProvider.ppc(shard), modelIds, clazz);
    }

    /**
     * Возвращает объекты, в которых заполнен данными только указанный интерфейс.
     * Если в выборку попадают объекты, которые не имплементируют указанный интерфейс,
     * то генерируется исключение (поэтому "strictly").
     * <p>
     * За подробным описанием обращайтесь к странице
     * https://doc.yandex-team.ru/direct-dev/concepts/docs/banner/concept.html#repo-read-strictly
     *
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     * @see #getStrictlyFullyFilled(DSLContext, Collection, Class)
     */
    public <M extends T> List<M> getStrictly(
            DSLContext dslContext,
            Collection<Long> modelIds,
            Class<M> clazz) {
        if (modelIds.isEmpty()) {
            return emptyList();
        }

        SelectQuery<Record> query = selectClassFields(dslContext, singletonList(clazz));
        typeSupportFacade.collectSelectJoinStep(query, clazz);
        getIdFilter(modelIds).apply(query);

        Result<Record> records = query.fetch();
        List<T> models = mapList(records, rec -> typeSupportFacade.getModelFromRecord(rec, clazz));
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models, clazz);

        List<M> filteredModels = StreamEx.of(models)
                .select(clazz)
                .toList();
        checkState(models.size() == filteredModels.size(),
                "some models selected by ids %s do not extend class %s", modelIds, clazz.getName());
        return filteredModels;
    }

    /**
     * @see #getStrictlyFullyFilled(DSLContext, Collection, Class)
     */
    public <M extends T> List<M> getStrictlyFullyFilled(int shard, Collection<Long> modelIds, Class<M> clazz) {
        return getStrictlyFullyFilled(dslContextProvider.ppc(shard), modelIds, clazz);
    }

    /**
     * Метод возвращает полностью заполненные объекты разных типов,
     * которые имплементируют указанный интерфейс.
     * <p>
     * Если в выборку попадают объекты, которые не имплементируют указанный интерфейс,
     * то генерируется исключение (поэтому "strictly").
     * <p>
     * За подробным описанием обращайтесь к странице
     * https://doc.yandex-team.ru/direct-dev/concepts/docs/banner/concept.html#repo-read-strictly-full
     *
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     * @see #getStrictly(DSLContext, Collection, Class)
     */
    public <M extends T> List<M> getStrictlyFullyFilled(DSLContext dslContext,
                                                        Collection<Long> modelIds,
                                                        Class<M> clazz) {
        List<T> models = getTyped(dslContext, modelIds);
        List<M> filteredModels = StreamEx.of(models)
                .select(clazz)
                .toList();
        checkState(models.size() == filteredModels.size(),
                "some models selected by ids %s do not extend class %s", modelIds, clazz.getName());
        return filteredModels;
    }

    protected SelectQuery<Record> getBaseQuery(DSLContext dslContext) {
        SelectQuery<Record> query = selectAllFields(dslContext);
        return typeSupportFacade.collectSelectJoinStep(query);
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    protected List<T> getTyped(DSLContext dslContext, Filter filter) {
        return getTyped(dslContext, filter, null);
    }

    /**
     * @see #getTyped(DSLContext, SelectQuery, Filter, LimitOffset)
     */
    protected List<T> getTyped(DSLContext dslContext, Filter filter,
                               @Nullable LimitOffset limitOffset) {
        return getTyped(dslContext, null, filter, limitOffset);
    }

    private SelectQuery<Record> selectAllFields(DSLContext dslContext) {
        return dslContext
                .select(allFields)
                .from(getBaseTable())
                .getQuery();
    }

    private SelectQuery<Record> selectClassFields(DSLContext dslContext, Collection<Class<? extends T>> classes) {
        return dslContext
                .select(typeSupportFacade.getModelFields(classes))
                .from(getBaseTable())
                .getQuery();
    }

    /**
     * Метод выбирает из базы и возвращает полностью заполненные объекты разных типов.
     * <p>
     * В момент вызова метода ему ничего не известно о типах запрашиваемых объектов,
     * поэтому в sql-запросе будут перечислены все поля и сджойнятся все таблицы
     * в отношении один-к-одному к основной таблице.
     * <p>
     * За подробным описанием обращайтесь к странице
     * https://docs.yandex-team.ru/direct-dev/dev/banner/concept#repo-read-typed
     *
     * @see #getStrictly(DSLContext, Collection, Class)
     * @see #getSafely(DSLContext, Filter, LimitOffset, Collection)
     * @see #getStrictlyFullyFilled(DSLContext, Collection, Class)
     */
    protected List<T> getTyped(DSLContext dslContext,
                               @Nullable SelectQuery<Record> query,
                               @Nullable Filter filter,
                               @Nullable LimitOffset limitOffset) {

        if (query == null) {
            query = getBaseQuery(dslContext);
        }
        if (filter != null) {
            if (filter.isEmpty()) {
                return emptyList();
            }
            filter.apply(query);
        }
        if (limitOffset != null) {
            query.addOffset(limitOffset.offset());
            query.addLimit(limitOffset.limit());
        }
        Result<Record> records = query.fetch();
        List<T> models = mapList(records, typeSupportFacade::getModelFromRecord);
        typeSupportFacade.enrichModelFromOtherTables(dslContext, models);
        return models;
    }
}
