package ru.yandex.direct.multitype.repository.container;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.ImmutableList;
import one.util.streamex.EntryStream;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Реализация контейнера для случая, когда определение того, что необходимо добавлять,
 * а что удалять, работает не на основе метода equals модели, а на основе equals/hashCode
 * ключа, извлекаемого из модели (детальнее про ключ - ниже).
 * <p>
 * Ключ работает в рамках одной родительской модели (например, баннера),
 * поэтому в нем не должен содержаться её id, см. {@link Builder}.
 *
 * @param <T> тип модели, отражающей запись в соответствующей таблице.
 * @see AddOrUpdateAndDeleteContainer
 */
@ParametersAreNonnullByDefault
public final class AddOrUpdateAndDeleteByKeyContainer<T> implements AddOrUpdateAndDeleteContainer<T> {

    private final List<T> rowsToDelete;
    private final List<T> rowsToAddOrUpdate;

    private AddOrUpdateAndDeleteByKeyContainer(List<T> rowsToDelete, List<T> rowsToAddOrUpdate) {
        this.rowsToDelete = ImmutableList.copyOf(rowsToDelete);
        this.rowsToAddOrUpdate = ImmutableList.copyOf(rowsToAddOrUpdate);
    }

    @Override
    public Collection<T> getRowsToDelete() {
        return rowsToDelete;
    }

    @Override
    public Collection<T> getRowsToAddOrUpdate() {
        return rowsToAddOrUpdate;
    }

    public static <T, K> Builder<T, K> builder(Function<T, K> keyMapper) {
        return new Builder<>(keyMapper);
    }

    public static class Builder<T, K> {

        @FunctionalInterface
        public interface SkipRowCondition<T, K> {
            boolean test(BiMap<K, T> oldRowsMap, K newKey, T newRow);
        }

        private final Function<T, K> keyMapper;
        private final List<T> rowsToDelete = new ArrayList<>();
        private final List<T> rowsToAddOrUpdate = new ArrayList<>();

        // по умолчанию все новые строки отправляем в addOrUpdate
        private SkipRowCondition<T, K> skipNewRowCondition = (oldRowsMap, newRowKey, newRow) -> Boolean.FALSE;

        /**
         * @param keyMapper функция извлечения логического ключа из строки таблицы
         *                  (под строкой понимается модель, отражающая строку),
         *                  с помощью которого определяется, какие строки необходимо удалить,
         *                  а какие добавить или обновить; в ключ не входит ключ родительской модели,
         *                  так как метод #addDiff всегда вызывается в контексте одной родительской модели.
         */
        public Builder(Function<T, K> keyMapper) {
            this.keyMapper = keyMapper;
        }

        public Builder<T, K> skipNewRowsByEquals() {
            return skipNewRowsWhen((oldRows, newRowKey, newRow) -> oldRows.inverse().containsKey(newRow));
        }

        public Builder<T, K> skipNewRowsByKeyEquals() {
            return skipNewRowsWhen((oldRows, newRowKey, newRow) -> oldRows.containsKey(newRowKey));
        }

        public Builder<T, K> skipNewRowsWhen(SkipRowCondition<T, K> skipNewRowCondition) {
            this.skipNewRowCondition = skipNewRowCondition;
            return this;
        }

        /**
         * Метод добавления изменений списка дочерних элементов.
         * Вызывается в контексте одной родительской модели (например, баннера).
         *
         * @param oldRows текущий список записей в базе (в виде моделей).
         * @param newRows список записей в базе, который должен получиться в результате обновления.
         */
        public Builder<T, K> addDiff(Collection<T> oldRows, Collection<T> newRows) {
            BiMap<K, T> keyToOldRows = HashBiMap.create(listToMap(oldRows, keyMapper));
            BiMap<K, T> keyToNewRows = HashBiMap.create(listToMap(newRows, keyMapper));

            List<T> localRowsToDelete = EntryStream.of(keyToOldRows)
                    .removeKeys(keyToNewRows::containsKey)
                    .values()
                    .toList();

            List<T> localRowsToAddOrUpdate = EntryStream.of(keyToNewRows)
                    .removeKeyValue((newRowKey, newRow) ->
                            skipNewRowCondition.test(keyToOldRows, newRowKey, newRow))
                    .values()
                    .toList();

            rowsToDelete.addAll(localRowsToDelete);
            rowsToAddOrUpdate.addAll(localRowsToAddOrUpdate);
            return this;
        }

        public AddOrUpdateAndDeleteByKeyContainer<T> build() {
            return new AddOrUpdateAndDeleteByKeyContainer<>(rowsToDelete, rowsToAddOrUpdate);
        }
    }
}
