package ru.yandex.direct.multitype.service.type.update;

import java.util.List;

import org.jooq.DSLContext;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.typesupport.TypeSupport;

/**
 * @param <T> - конечный тип, который обрабатывается репозиторием
 * @param <A> - контейнер, который передается в методы обновления
 * @param <B> - контейнер, который передается в методы обновления
 */
public interface UpdateOperationTypeSupport<T extends ModelWithId, A, B, C> extends TypeSupport<T> {

    /**
     * Выполнить действия сразу после того, как modelChanges провалидированы
     */
    void onModelChangesValidated(A updateContainer, List<ModelChanges<T>> validModelChanges);

    /**
     * Выполнить действия сразу после того, как модели загружены и к ним применены изменения
     */
    void onChangesApplied(A updateContainer, List<AppliedChanges<T>> appliedChanges);

    /**
     * Выполнить действия после валидации моделей с примененными изменениями
     */
    void onAppliedChangesValidated(A updateContainer, List<AppliedChanges<T>> appliedChanges);

    /**
     * Выполнить действия до создания
     */
    void beforeExecution(A updateContainer, List<AppliedChanges<T>> appliedChanges);

    /**
     * Добавить данные в контейнер, для последующего применения действий
     * Например для сброса статуса синхронизации с БК для групп и баннеров
     * Или добавления в очередь на отправку нотификаций в Баланс
     */
    void addToAdditionalActionsContainer(B additionalActionsContainer,
                                         A updateContainer,
                                         List<AppliedChanges<T>> appliedChanges);

    /**
     * Изменяет статусы модерации для сввязанных объектов, таких как креативы, сайтлинки
     */
    void beforeExecutionInTransaction(DSLContext dsl,
                                      B additionalActionsContainer,
                                      A updateContainer,
                                      List<AppliedChanges<T>> appliedChanges);

    /**
     * Обновить в рамках транзакции смежные сущности
     */
    void updateRelatedEntitiesInTransaction(
            DSLContext dslContext,
            A updateContainer,
            List<AppliedChanges<T>> appliedChanges);

    /**
     * Обновить не в рамках транзакции смежные сущности
     */
    void updateRelatedEntitiesOutOfTransaction(A updateContainer,
                                               List<AppliedChanges<T>> appliedChanges);

    /**
     * Обновить не в рамках транзакции смежные сущности с учетом modelChanges
     */
    void updateRelatedEntitiesOutOfTransactionWithModelChanges(A updateContainer,
                                                               List<ModelChanges<T>> modelChanges,
                                                               List<AppliedChanges<T>> appliedChanges);

    /**
     * Выполнить действия после создания
     */
    void afterExecution(A updateContainer, List<AppliedChanges<T>> appliedChanges);

    /**
     * Будет проверять, есть ли в рамках данного интерфейса достаточно значительные изменения для переотправки
     * объекта в модерацию
     */
    boolean needModeration(C container, AppliedChanges<T> appliedChanges);

    /**
     * Будет проверять, есть ли в рамках данного интерфейса достаточно значительные изменения для переотправки
     * объекта в БК
     */
    boolean needBsResync(AppliedChanges<T> appliedChanges);

    /**
     * Будет проверять, необходимость обновления поля LastChange
     */
    boolean needLastChangeReset(AppliedChanges<T> appliedChanges);

}
