package ru.yandex.direct.multitype.service.type.update;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.typesupport.TypeSupport;
import ru.yandex.direct.multitype.typesupport.TypeSupportAffectionHelper;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public abstract class UpdateOperationTypeSupportFacade
        <T extends ModelWithId, A extends UpdateOperationContainer<T>, B, C> {
    private final List<? extends UpdateOperationTypeSupport<? extends T, A, B, C>> supports;
    private final TypeSupportAffectionHelper<T> typeSupportAffectionHelper;

    public UpdateOperationTypeSupportFacade(List<? extends UpdateOperationTypeSupport<? extends T, A, B, C>> supports,
                                            Set<Class<? extends T>> whiteListSupportTypeClass) {
        this.supports = supports;
        this.typeSupportAffectionHelper = new TypeSupportAffectionHelper<>(whiteListSupportTypeClass);
    }

    public Map<? extends UpdateOperationTypeSupport<? extends T, A, B, C>,
            ? extends List<? extends ModelChanges<? extends T>>> getModelChangesGroupedByTypeSupports(
            A updateContainer,
            Collection<? extends ModelChanges<? extends T>> modelChanges) {
        return StreamEx.of(supports)
                .mapToEntry(TypeSupport::getTypeClass)
                .mapValues(supportClass -> typeSupportAffectionHelper.selectModelChangesThatAffectSupport(
                        updateContainer, modelChanges, supportClass))
                .removeValues(List::isEmpty)
                .toMap();
    }

    public Map<? extends UpdateOperationTypeSupport<? extends T, A, B, C>,
            ? extends List<? extends AppliedChanges<? extends T>>>
    getAppliedChangesGroupedByTypeSupports(
            Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        return StreamEx.of(supports)
                .mapToEntry(TypeSupport::getTypeClass)
                .mapValues(supportClass -> typeSupportAffectionHelper.selectAppliedChangesThatAffectSupport(
                        appliedChanges, supportClass))
                .removeValues(List::isEmpty)
                .toMap();
    }

    public void beforeExecution(List<? extends AppliedChanges<? extends T>> appliedChanges, A updateContainer) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, typedChanges) ->
                beforeExecution(support, typedChanges, updateContainer));
    }

    private <M extends T> void beforeExecution(
            UpdateOperationTypeSupport<M, A, B, C> support,
            List<? extends AppliedChanges<? extends T>> typedAppliedChanges,
            A updateContainer) {
        List<AppliedChanges<M>> appliedChanges = castAppliedChanges(support.getTypeClass(), typedAppliedChanges);
        support.beforeExecution(updateContainer, appliedChanges);
    }

    public void afterExecution(List<? extends AppliedChanges<? extends T>> appliedChanges, A updateContainer) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, typedChanges) ->
                afterExecution(support, typedChanges, updateContainer));
    }

    private <M extends T> void afterExecution(
            UpdateOperationTypeSupport<M, A, B, C> support,
            List<? extends AppliedChanges<? extends T>> typedAppliedChanges,
            A updateContainer) {
        List<AppliedChanges<M>> appliedChanges = castAppliedChanges(support.getTypeClass(), typedAppliedChanges);
        support.afterExecution(updateContainer, appliedChanges);
    }

    public void updateRelatedEntitiesInTransaction(
            DSLContext context,
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {

        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, typedChanges) ->
                updateRelatedEntitiesInTransaction(context, support, updateContainer, typedChanges));
    }

    private <M extends T> void updateRelatedEntitiesInTransaction(
            DSLContext context,
            UpdateOperationTypeSupport<M, A, B, C> support,
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<AppliedChanges<M>> appliedChangesOfCertainType = castAppliedChanges(support.getTypeClass(),
                appliedChanges);
        support.updateRelatedEntitiesInTransaction(context, updateContainer, appliedChangesOfCertainType);
    }

    public void updateRelatedEntitiesOutOfTransaction(
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {

        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, typedAppliedChanges) ->
                updateRelatedEntitiesOutOfTransaction(support, updateContainer, typedAppliedChanges));
    }

    private <M extends T> void updateRelatedEntitiesOutOfTransaction(
            UpdateOperationTypeSupport<M, A, B, C> support,
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<AppliedChanges<M>> appliedChangesOfCertainType = castAppliedChanges(support.getTypeClass(),
                appliedChanges);
        support.updateRelatedEntitiesOutOfTransaction(updateContainer, appliedChangesOfCertainType);
    }

    public void updateRelatedEntitiesOutOfTransactionWithModelChanges(
            A updateContainer,
            List<? extends ModelChanges<? extends T>> modelChanges,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        for (var support : supports) {
            var applicableModelChanges = typeSupportAffectionHelper.selectModelChangesThatAffectSupport(
                    updateContainer, modelChanges, support.getTypeClass());
            Set<Long> modelIds = listToSet(applicableModelChanges, ModelChanges::getId);
            // для вычисления applicableAppliedChanges не подходит стандартная процедура из typeSupportAffectionHelper
            // потому что дополнительно нужно всегда включать модель, если она попала в applicableModelChanges
            var applicableAppliedChanges = appliedChanges.stream()
                    .filter(ac -> modelIds.contains(ac.getModel().getId())
                            || typeSupportAffectionHelper.isAppliedChangesAffectsSupport(ac, support.getTypeClass()))
                    .collect(Collectors.toList());
            updateRelatedEntitiesOutOfTransactionWithModelChanges(
                    updateContainer,
                    support,
                    applicableModelChanges,
                    applicableAppliedChanges);
        }
    }

    private <M extends T> void updateRelatedEntitiesOutOfTransactionWithModelChanges(
            A updateContainer,
            UpdateOperationTypeSupport<M, A, B, C> support,
            @Nullable List<? extends ModelChanges<? extends T>> modelChangesGroupedByTypeSupports,
            @Nullable List<? extends AppliedChanges<? extends T>> appliedChangesGroupedByTypeSupports) {
        if (appliedChangesGroupedByTypeSupports != null) {
            @SuppressWarnings("ConstantConditions")
            List<ModelChanges<M>> modelChanges = castModelChanges(support.getTypeClass(),
                    modelChangesGroupedByTypeSupports);
            List<AppliedChanges<M>> appliedChanges = castAppliedChanges(support.getTypeClass(),
                    appliedChangesGroupedByTypeSupports);
            support.updateRelatedEntitiesOutOfTransactionWithModelChanges(updateContainer, modelChanges,
                    appliedChanges);
        }
    }

    public void onModelChangesValidated(A updateContainer,
                                        Collection<? extends ModelChanges<? extends T>> modelChanges) {
        getModelChangesGroupedByTypeSupports(updateContainer, modelChanges).forEach((support, supportedModelChanges) ->
                onModelChangesValidated(supportedModelChanges, updateContainer, support));
    }

    private <M extends T> void onModelChangesValidated(List<? extends ModelChanges<? extends T>> modelChanges,
                                                       A updateContainer,
                                                       UpdateOperationTypeSupport<M, A, B, C> support) {
        List<ModelChanges<M>> typedModelChanges = castModelChanges(support.getTypeClass(), modelChanges);
        support.onModelChangesValidated(updateContainer, typedModelChanges);
    }

    public void onChangesApplied(A updateContainer, List<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, supportedAppliedChanges) ->
                onChangesApplied(supportedAppliedChanges, updateContainer, support));
    }

    private <M extends T> void onChangesApplied(List<? extends AppliedChanges<? extends T>> appliedChanges,
                                                A updateContainer,
                                                UpdateOperationTypeSupport<M, A, B, C> support) {
        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        support.onChangesApplied(updateContainer, typedAppliedChanges);
    }

    public void onAppliedChangesValidated(A updateContainer,
                                          List<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, supportedAppliedChanges) ->
                onAppliedChangesValidated(support, updateContainer, supportedAppliedChanges));
    }

    private <M extends T> void onAppliedChangesValidated(
            UpdateOperationTypeSupport<M, A, B, C> support,
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        support.onAppliedChangesValidated(updateContainer, typedAppliedChanges);
    }

    public void addToAdditionalActionsContainer(
            B additionalContainer,
            A updateContainer,
            Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, changes) ->
                addToAdditionalActionsContainer(additionalContainer, updateContainer, changes, support));
    }

    private <M extends T> void addToAdditionalActionsContainer(
            B additionalContainer,
            A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges,
            UpdateOperationTypeSupport<M, A, B, C> support) {
        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        support.addToAdditionalActionsContainer(additionalContainer, updateContainer, typedAppliedChanges);
    }

    public void beforeExecutionInTransaction(
            DSLContext dsl,
            B additionalActionsContainer, A updateContainer,
            Collection<? extends AppliedChanges<? extends T>> appliedChanges) {
        getAppliedChangesGroupedByTypeSupports(appliedChanges).forEach((support, changes) ->
                beforeExecutionInTransaction(dsl, additionalActionsContainer, updateContainer, changes, support));
    }

    private <M extends T> void beforeExecutionInTransaction(
            DSLContext dsl,
            B additionalActionsContainer, A updateContainer,
            List<? extends AppliedChanges<? extends T>> appliedChanges,
            UpdateOperationTypeSupport<M, A, B, C> support) {
        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        support.beforeExecutionInTransaction(dsl, additionalActionsContainer, updateContainer, typedAppliedChanges);
    }

    public Set<Long> getNeedModerationIds(
            C container,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {
        return EntryStream.of(getAppliedChangesGroupedByTypeSupports(appliedChanges))
                .mapKeyValue((support, ac) -> getNeedModerationIds(support, container, ac))
                .flatMap(Collection::stream)
                .toSet();
    }

    private <M extends T> Set<Long> getNeedModerationIds(
            UpdateOperationTypeSupport<M, A, B, C> support,
            C container,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {

        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        return filterAndMapToSet(typedAppliedChanges, ac -> support.needModeration(container, ac),
                x -> x.getModel().getId());
    }

    public Set<Long> getNeedBsResyncIds(List<? extends AppliedChanges<? extends T>> appliedChanges) {
        return EntryStream.of(getAppliedChangesGroupedByTypeSupports(appliedChanges))
                .mapKeyValue(this::getNeedBsResyncIds)
                .flatMap(Collection::stream)
                .toSet();
    }

    private <M extends T> Set<Long> getNeedBsResyncIds(
            UpdateOperationTypeSupport<M, A, B, C> support,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {

        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        return filterAndMapToSet(typedAppliedChanges, support::needBsResync, x -> x.getModel().getId());
    }

    public Set<Long> getNeedLastChangeResetIds(List<? extends AppliedChanges<? extends T>> appliedChanges) {
        return EntryStream.of(getAppliedChangesGroupedByTypeSupports(appliedChanges))
                .mapKeyValue(this::getNeedLastChangeResetIds)
                .flatMap(Collection::stream)
                .toSet();
    }

    private <M extends T> Set<Long> getNeedLastChangeResetIds(
            UpdateOperationTypeSupport<M, A, B, C> support,
            List<? extends AppliedChanges<? extends T>> appliedChanges) {

        List<AppliedChanges<M>> typedAppliedChanges = castAppliedChanges(support.getTypeClass(), appliedChanges);
        return filterAndMapToSet(typedAppliedChanges, support::needLastChangeReset, x -> x.getModel().getId());
    }

    private <M extends T> List<AppliedChanges<M>> castAppliedChanges(Class<M> typeClass,
                                                                     List<? extends AppliedChanges<? extends T>> typedAppliedChanges) {
        return mapList(typedAppliedChanges, changes -> changes.castModelUp(typeClass));
    }

    public <M extends T> List<ModelChanges<M>> castModelChanges(Class<M> typeClass,
                                                                 List<? extends ModelChanges<? extends T>> typedModelChanges) {
        return mapList(typedModelChanges, changes -> changes.castModel(typeClass));
    }

    protected List<? extends UpdateOperationTypeSupport<? extends T, A, B, C>> getSupports() {
        return supports;
    }

    protected TypeSupportAffectionHelper<T> getTypeSupportAffectionHelper() {
        return typeSupportAffectionHelper;
    }
}
