package ru.yandex.direct.multitype.service.validation.type;

import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationContainer;
import ru.yandex.direct.multitype.typesupport.TypeSupportAffectionHelper;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getOrCreateSubValidationResultWithoutCast;

@ParametersAreNonnullByDefault
public class ValidationTypeSupportUtils {

    public static <T, D, M extends T> ValidationResult<List<M>, D> select(
            ValidationResult<? extends List<? extends T>, D> vr,
            Class<M> clazz) {
        List<Integer> indexes = selectIndexes(vr, clazz);
        List<M> values = StreamEx.of(vr.getValue())
                .select(clazz)
                .toList();
        return buildSubValidationResult(vr, indexes, values);
    }

    public static <T, D, M extends T> ValidationResult<List<M>, D> buildSubValidationResult(
            ValidationResult<? extends List<? extends T>, D> vr,
            List<Integer> indexes,
            List<M> values) {
        @SuppressWarnings("unchecked")
        List<ValidationResult<?, D>> subResults = (List<ValidationResult<?, D>>) getSubResults(vr, indexes);
        Map<PathNode, ValidationResult<?, D>> subResultsMap = getSubResultsMap(subResults);
        return new ValidationResult<>(values, vr.getErrors(), vr.getWarnings(), subResultsMap);
    }

    @Nullable
    @SuppressWarnings("unchecked")
    public static <T extends Model, M extends T> Map<Integer, AppliedChanges<M>> selectAppliedChanges(
            List<Integer> sortedIndexes,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {

        Map<Integer, Integer> oldIndexToNewIndexMap = EntryStream.of(sortedIndexes)
                .invert()
                .toMap();

        return (Map<Integer, AppliedChanges<M>>)
                EntryStream.of(appliedChangesForValidModelChanges)
                        .removeKeys(not(oldIndexToNewIndexMap::containsKey))
                        .mapKeys(oldIndexToNewIndexMap::get)
                        .toMap();
    }

    public static <T, D, M extends T> List<Integer> selectIndexes(ValidationResult<? extends List<? extends T>, D> vr,
                                                                  Class<M> clazz) {
        return EntryStream.of(vr.getValue())
                .selectValues(clazz)
                .keys()
                .sorted()
                .toList();
    }

    public static <T, D, M extends T> List<Integer> filterIndexes(ValidationResult<? extends List<? extends T>, D> vr,
                                                                  Class<M> clazz,
                                                                  BiPredicate<Integer, T> predicate) {
        return filterValidationResult(vr, clazz, predicate)
                .keys()
                .toList();
    }

    public static <T, D, M extends T> List<M> filterValues(ValidationResult<? extends List<? extends T>, D> vr,
                                                           Class<M> clazz,
                                                           BiPredicate<Integer, T> predicate) {
        return filterValidationResult(vr, clazz, predicate)
                .values()
                .toList();
    }

    private static <T, D, M extends T> EntryStream<Integer, M> filterValidationResult(
            ValidationResult<? extends List<? extends T>, D> vr,
            Class<M> clazz,
            BiPredicate<Integer, T> predicate) {
        return EntryStream.of(vr.getValue())
                .selectValues(clazz)
                .filterKeyValue(predicate)
                .sortedBy(Map.Entry::getKey);
    }

    private static <T, D> List<? extends ValidationResult<?, D>> getSubResults(
            ValidationResult<? extends List<? extends T>, D> vr,
            List<Integer> indexes) {
        return StreamEx.of(indexes)
                .sorted()
                .map(PathNode.Index::new)
                .map(index -> vr.getOrCreateSubValidationResultWithoutCast(index, vr.getValue().get(index.getIndex())))
                .toList();
    }

    public static <T extends ModelWithId, D> ValidationResult<List<ModelChanges<T>>, D>
    filterValidSubResults(ValidationResult<List<ModelChanges<T>>, D> vr) {
        return filterSubResults(vr, subVr -> !subVr.hasAnyErrors());
    }

    public static <T extends ModelWithId, D> ValidationResult<List<ModelChanges<T>>, D>
    filterSubResults(ValidationResult<List<ModelChanges<T>>, D> vr,
                     Predicate<ValidationResult<?, D>> predicate) {
        List<ValidationResult<?, D>> allSubValidationResults = getSubValidationResults(vr);

        List<Integer> indexes = EntryStream.of(allSubValidationResults)
                .filterValues(predicate)
                .keys()
                .toList();

        List<ModelChanges<T>> values = mapList(indexes, vr.getValue()::get);

        List<ValidationResult<?, D>> subValidationResultsByPredicate = getSubValidationResults(vr, indexes);
        Map<PathNode, ValidationResult<?, D>> subResultsMap = getSubResultsMap(subValidationResultsByPredicate);

        return new ValidationResult<>(values, vr.getErrors(), vr.getWarnings(), subResultsMap);
    }

    /**
     * Фильтрует ValidationResult<ModelChanges>. Включает модель в результат, только если typeSupportAffectionHelper
     * говорит, что этот modelChanges affects supportClass.
     */
    public static <T extends ModelWithId, D, M extends T> ValidationResult<List<ModelChanges<M>>, D> selectModelChanges(
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, D> vr,
            UpdateOperationContainer<?> updateOperationContainer,
            Class<M> supportClass,
            TypeSupportAffectionHelper<T> typeSupportAffectionHelper) {
        Predicate<ModelChanges<?>> includeModelChanges = mc ->
                typeSupportAffectionHelper.isModelChangesAffectsSupport(updateOperationContainer, mc, supportClass);

        List<Integer> indexes = EntryStream.of(vr.getValue())
                .filterValues(includeModelChanges)
                .keys()
                .toList();

        List<ModelChanges<M>> values = StreamEx.of(vr.getValue())
                .filter(includeModelChanges)
                // можем так сделать, т.к. знаем, что конечный класс имплементить запрашиваемый supportClass
                // это обязательно должен был проверить предыдущий вызов includeModelChanges
                .map(mc -> mc.castModel(supportClass))
                .toList();

        List<ValidationResult<?, D>> subResults = getSubValidationResults(vr, indexes);
        Map<PathNode, ValidationResult<?, D>> subResultsMap = getSubResultsMap(subResults);

        return new ValidationResult<>(values, vr.getErrors(), vr.getWarnings(), subResultsMap);
    }

    private static <D> Map<PathNode, ValidationResult<?, D>> getSubResultsMap(List<ValidationResult<?, D>> subResults) {
        return EntryStream.of(subResults)
                .mapKeys(PathNode.Index::new)
                .mapKeys(index -> (PathNode) index)
                .toMap();
    }

    private static <T extends List<?>, D> List<ValidationResult<?, D>> getSubValidationResults(
            ValidationResult<T, D> vr, List<Integer> indexes) {
        return indexes.stream()
                .map(index -> getOrCreateSubValidationResultWithoutCast(vr, index))
                .collect(Collectors.toList());
    }

    private static <T extends ModelWithId, D> List<ValidationResult<?, D>> getSubValidationResults(
            ValidationResult<List<ModelChanges<T>>, D> vr) {
        return IntStreamEx.range(vr.getValue().size())
                .boxed()
                .map(PathNode.Index::new)
                .map(index -> vr.getOrCreateSubValidationResultWithoutCast(index, vr.getValue().get(index.getIndex())))
                .collect(Collectors.toList());
    }

    public static <T extends ModelWithId> void addDefectOnField(ValidationResult<List<ModelChanges<T>>,
            Defect> vr, ModelProperty property, Integer index, Defect defect) {
        vr.getOrCreateSubValidationResult(index(index), vr.getValue().get(index))
                .getOrCreateSubValidationResult(field(property), property)
                .addError(defect);
    }

    public static <M extends ModelWithId> void forEachModelChangesAddDefectIfFieldChanged(
            ValidationResult<List<ModelChanges<M>>, Defect> vr, ModelProperty<?, ?> modelProperty) {
        List<ModelChanges<M>> modelChanges = vr.getValue();
        for (int i = 0; i < modelChanges.size(); i++) {
            if (modelChanges.get(i).isPropChanged(modelProperty)) {
                addDefectOnField(vr, modelProperty, i, new Defect<>(DefectIds.FORBIDDEN_TO_CHANGE));
            }
        }
    }

}
