package ru.yandex.direct.multitype.service.validation.type.update;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationContainer;
import ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils;
import ru.yandex.direct.multitype.typesupport.TypeSupportAffectionHelper;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.buildSubValidationResult;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterIndexes;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValidSubResults;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValues;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.selectAppliedChanges;

@ParametersAreNonnullByDefault
public class UpdateValidationTypeSupportFacade<T extends ModelWithId, C extends UpdateOperationContainer<?>> {

    private final List<? extends UpdateValidationTypeSupport<? extends T, C>> supports;
    private final TypeSupportAffectionHelper<T> typeSupportAffectionHelper;

    public UpdateValidationTypeSupportFacade(List<? extends UpdateValidationTypeSupport<? extends T, C>> supports,
                                             Set<Class<? extends T>> whiteListSupportTypeClass) {
        this.supports = supports;
        this.typeSupportAffectionHelper = new TypeSupportAffectionHelper<>(whiteListSupportTypeClass);
    }

    public void preValidate(C container, ValidationResult<? extends List<?
            extends ModelChanges<? extends T>>, Defect> vr) {
        supports.forEach(support -> preValidateBySupport(container, support, vr));
    }

    private <T2 extends T> void preValidateBySupport(
            C container, UpdateValidationTypeSupport<T2, C> support,
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, Defect> vr) {
        var subVr = ValidationTypeSupportUtils.selectModelChanges(vr, container,
                support.getTypeClass(), typeSupportAffectionHelper);

        if (!subVr.getValue().isEmpty()) {
            support.preValidate(container, subVr);
        }
    }

    public <T2 extends T> void validateBeforeApply(C container,
                                                   ValidationResult<List<ModelChanges<T2>>, Defect> vr,
                                                   Map<Long, ? extends T> unmodifiedModels) {
        ValidationResult<List<ModelChanges<T2>>, Defect> vrWithValidSubResults = filterValidSubResults(vr);
        supports.forEach(support -> validateBeforeApplyBySupport(container, support, vrWithValidSubResults,
                unmodifiedModels));
        vrWithValidSubResults.getErrors().forEach(error -> {
            if (!vr.getErrors().contains(error)) {
                vr.addError(error);
            }
        });
    }

    private <T2 extends T, T3 extends T> void validateBeforeApplyBySupport(
            C container,
            UpdateValidationTypeSupport<T2, C> support,
            ValidationResult<List<ModelChanges<T3>>, Defect> vr,
            Map<Long, ? extends T> unmodifiedModels
    ) {
        var subVr = ValidationTypeSupportUtils.selectModelChanges(vr, container,
                support.getTypeClass(), typeSupportAffectionHelper);

        if (!subVr.getValue().isEmpty()) {
            Map<Long, T2> typedUnmodifiedModels = EntryStream.of(unmodifiedModels)
                    .selectValues(support.getTypeClass())
                    .toMap();

            support.validateBeforeApply(container, subVr, typedUnmodifiedModels);

            // Перебросить ошибки уровня операции в validationResult. Ошибки уровня операции содержат ошибки на всю
            // коллекцию обновляемых стратегий. Например, ошибка превышения лимита общего количества стратегий.
            subVr.getErrors().forEach(error -> {
                if (!vr.getErrors().contains(error)) {
                    vr.addError(error);
                }
            });
        }
    }

    public void validate(
            C container,
            ValidationResult<? extends List<? extends T>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {
        supports.forEach(support -> {
            var applicableAppliedChanges = typeSupportAffectionHelper
                    .selectAppliedChangesThatAffectSupport(appliedChangesForValidModelChanges,
                            support.getTypeClass());
            if (applicableAppliedChanges.size() > 0) {
                validate(container, support, vr, applicableAppliedChanges);
            }
        });
    }

    private <T2 extends T> void validate(
            C container,
            UpdateValidationTypeSupport<T2, C> support,
            ValidationResult<? extends List<? extends T>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends T>> appliedChangesForValidModelChanges) {
        BiPredicate<Integer, T> isValidModel =
                (index, x) -> appliedChangesForValidModelChanges.containsKey(index);
        List<Integer> indexes = filterIndexes(vr, support.getTypeClass(), isValidModel);
        List<T2> values = filterValues(vr, support.getTypeClass(), isValidModel);
        ValidationResult<List<T2>, Defect> subVr = buildSubValidationResult(vr, indexes, values);
        Map<Integer, AppliedChanges<T2>> subAppliedChanges = selectAppliedChanges(indexes,
                appliedChangesForValidModelChanges);
        if (!subVr.getValue().isEmpty()) {
            support.validate(container, subVr, subAppliedChanges);
        }
    }

}
