package ru.yandex.direct.multitype.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;

import static java.util.stream.Collectors.toList;

@ParametersAreNonnullByDefault
public class TypeFilteringUtils {

    private TypeFilteringUtils() {
    }

    /**
     * Получить коллекцию, сформированную из исходной, и содержащую только
     * инстансы указанного класса или его наследников.
     *
     * @param models исходная коллекция моделей
     * @param cls    класс, инстансы которого попадут в результат
     * @param <T>    тип моделей
     * @return Список инстансов указанного класса или его наследников.
     */
    public static <T> List<T> filterModelsOfClass(Collection<? super T> models, Class<T> cls) {
        return models.stream()
                .filter(b -> cls.isAssignableFrom(b.getClass()))
                .map(cls::cast)
                .collect(toList());
    }

    /**
     * Получить коллекцию изменений ({@link AppliedChanges}),
     * сформированную из исходной, и содержащую только
     * изменения моделей-инстансов указанного класса или его наследников.
     *
     * @param appliedChangesCollection исходная коллекция изменений моделей
     * @param cls                      класс, изменений инстансов которого попадут в результат
     * @param <M>                      исходный тип моделей
     * @param <S>                      тип моделей в результирующей коллекции
     * @return Список изменений инстансов указанного класса или его наследников.
     * @see AppliedChanges
     */
    @SuppressWarnings("unchecked")
    public static <M extends Model, S extends M> List<AppliedChanges<S>> filterAppliedChangesForModelClass(
            Collection<AppliedChanges<M>> appliedChangesCollection, Class<S> cls) {
        return appliedChangesCollection.stream()
                .filter(ac -> cls.isAssignableFrom(ac.getModel().getClass()))
                .map(ac -> (AppliedChanges<S>) ac)
                .collect(toList());
    }
}
