package ru.yandex.direct.mysqlcompression;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Это такой ByteArrayOutputStream, в котором можно подглядеть последний записанный туда байт
 * <p>
 * Текущая реализация большую часть работы делегирует объекту ByteArrayOutputStream, который
 * она сама же и создаёт.
 *
 * @see ByteArrayOutputStream
 */
@ParametersAreNonnullByDefault
class LastByteTrackingByteArrayOutputStream extends OutputStream {
    private final ByteArrayOutputStream delegate;
    private byte lastByte;

    LastByteTrackingByteArrayOutputStream() {
        delegate = new ByteArrayOutputStream();
    }

    @Override
    public void write(int b) {
        synchronized (this) {
            lastByte = (byte) b;
            delegate.write(b);
        }
    }

    @Override
    public void write(byte[] b, int off, int len) {
        if (b == null) {
            throw new NullPointerException();
        } else if ((off < 0) || (off > b.length) || (len < 0) ||
                ((off + len) > b.length) || ((off + len) < 0)) {
            throw new IndexOutOfBoundsException();
        } else if (len == 0) {
            return;
        }

        synchronized (this) {
            lastByte = b[off + len - 1];
            delegate.write(b, off, len);
        }
    }

    public byte[] toByteArray() {
        return delegate.toByteArray();
    }

    public synchronized byte lastByte() {
        if (delegate.size() == 0) {
            throw new IllegalStateException();
        }

        return lastByte;
    }
}
