package ru.yandex.direct.mysqlcompression;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Класс из статических методов, которые сжимают и разжимают данные алгоритмами,
 * совместимыми с функциями compress() и uncompress() в MySQL.
 * <p>
 * Алгоритм там такой: {@see https://dev.mysql.com/doc/refman/5.7/en/encryption-functions.html#function_compress}
 * <p>
 * tl;dr:
 * <p>
 * Пустой массив данных сжимать не надо: compress() вернёт пустой массив.
 * <p>
 * Непустой массив при сжатии превращается в массив, в котором такое:
 * 1). 4 байта, в которых в low-endian order записан размер исходного массива
 * 2). сколько нужно байтов с исходными данными, пожатыми алгоритмом zlib deflate
 * 3). 0 или 1 байт: если последний символ пожатых данных из п. 2 пробел, ещё точка; если не пробел, ничего
 */
@ParametersAreNonnullByDefault
public class MysqlCompression {
    private MysqlCompression() {
    }

    public static byte[] compress(byte[] data) {
        if (data.length == 0) {
            return new byte[0];
        }

        LastByteTrackingByteArrayOutputStream out = new LastByteTrackingByteArrayOutputStream();

        // записать число в формате little-endian
        int remainingLengthBytes = data.length;
        for (int i = 0; i < 4; i++) {
            out.write(remainingLengthBytes % 256);
            remainingLengthBytes = remainingLengthBytes / 256;
        }

        // записать сжатые данные
        try (DeflaterOutputStream deflaterOut = new DeflaterOutputStream(out)) {
            deflaterOut.write(data);
        } catch (IOException e) {
            throw new MysqlCompressionException(e);
        }

        // если последний символ пробел, дописать ещё точку в конце
        if (out.lastByte() == ' ') {
            out.write('.');
        }

        return out.toByteArray();
    }

    /**
     * @throws MysqlCompressionException, если формат сжатых данных неправильный
     */
    public static byte[] uncompress(byte[] data) {
        if (data.length == 0) {
            return new byte[0];
        }

        try (ByteArrayInputStream in = new ByteArrayInputStream(data)) {
            byte[] packedUncompressedLength = new byte[4];
            int read = in.read(packedUncompressedLength);
            if (read != 4) {
                throw new MysqlCompressionException("failed to read 4 bytes with source length");
            }

            int uncompressedLength = 0;
            for (int i = 0; i < 4; i++) {
                int b = packedUncompressedLength[4 - i - 1];
                if (b < 0) {
                    b += 256;
                }

                uncompressedLength = 256 * uncompressedLength + b;
            }

            try (InflaterInputStream inflaterIn = new InflaterInputStream(in);
                 ByteArrayOutputStream resultStream = new ByteArrayOutputStream()) {
                byte[] buffer = new byte[1024];
                while (true) {
                    int readBytes = inflaterIn.read(buffer);
                    if (readBytes == -1) {
                        break;
                    }
                    resultStream.write(buffer, 0, readBytes);
                }

                byte[] result = resultStream.toByteArray();
                if (result.length != uncompressedLength) {
                    throw new MysqlCompressionException(String.format(
                            "unpacked data length doesn't match: unpacked %d bytes, expected %d bytes",
                            result.length, uncompressedLength));
                }

                return result;
            }
        } catch (IOException e) {
            throw new MysqlCompressionException(e);
        }
    }

}
