package ru.yandex.direct.mysql;

import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Base64;
import java.util.BitSet;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.github.shyiko.mysql.binlog.event.deserialization.json.JsonBinary;
import com.google.common.primitives.UnsignedBytes;
import com.google.common.primitives.UnsignedInts;
import com.google.common.primitives.UnsignedLongs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.util.Mysql57CompatibleJsonFormatter;
import ru.yandex.direct.utils.DateTimeUtils;

public class MySQLColumnType {
    private static final Logger logger = LoggerFactory.getLogger(MySQLColumnType.class);
    private static final String DEFAULT_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    private static final DateTimeFormatter UTC_DATE_TIME_FORMATTER = DateTimeFormatter
            .ofPattern(DEFAULT_DATETIME_FORMAT)
            .withZone(ZoneId.of("UTC"));

    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter
            .ofPattern(DEFAULT_DATETIME_FORMAT)
            .withZone(ZoneId.of("Europe/Moscow"));

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    static {
        OBJECT_MAPPER.setDefaultPrettyPrinter(new Mysql57CompatibleJsonFormatter.MysqlPrettyPrinter());
        OBJECT_MAPPER.enable(SerializationFeature.INDENT_OUTPUT);
    }

    /**
     * Кеширует строчки из схемы mysql в распарсенные объекты
     * <p>
     * Раньше здесь был гугловый кеш, но это место слишком горячее и потоки надолго зависают в drainRecencyQueue
     * <p>
     * Отсутствие ограничений на размер не страшно, т.к. в схемах не так много различных типов
     */
    private static ConcurrentMap<String, MySQLColumnType> cachedColumnTypes = new ConcurrentHashMap<>();

    private final MySQLDataType dataType;
    private final int width;
    private final int precision;
    private final List<String> values;
    private final boolean isUnsigned;

    /**
     * Кеширует строки дефолтных значений в java объекты
     * <p>
     * Раньше здесь был гугловый кеш, но это место слишком горячее и потоки надолго зависают в drainRecencyQueue
     * <p>
     * Отсутствие ограничений на размер не страшно, т.к. в схемах не так много различных дефолтных значений
     */
    private final ConcurrentMap<String, Serializable> cachedDefaultValues = new ConcurrentHashMap<>();

    /**
     * Parses MySQL column_type text
     */
    public MySQLColumnType(String text) {
        Parser parser = new Parser(text);
        Token token = parser.nextToken();
        if (token == null) {
            throw new IllegalArgumentException("Missing data type: " + text);
        }
        dataType = MySQLDataType.byName(token.getText());
        values = new ArrayList<>();
        int width = 0;
        int precision = 0;
        token = parser.nextToken();
        if (token != null && "(".equals(token.getText())) {
            // we have some parameters
            int index = 0;
            boolean finished = false;
            boolean expectComma = false;
            while ((token = parser.nextToken()) != null) {
                if (token.isQuotedString()) {
                    if (expectComma) {
                        throw new IllegalArgumentException("Missing comma between parameters: " + text);
                    }
                    values.add(token.getText());
                    expectComma = true;
                    continue;
                }
                if (")".equals(token.getText())) {
                    finished = true;
                    break;
                }
                if (",".equals(token.getText())) {
                    if (expectComma) {
                        expectComma = false;
                        continue;
                    }
                    throw new IllegalArgumentException("Unexpected comma in parameters: " + text);
                }
                if (expectComma) {
                    throw new IllegalArgumentException("Missing comma between parameters: " + text);
                }
                // Unquoted parameters must be integers
                int value;
                try {
                    value = Integer.parseInt(token.getText());
                } catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Cannot parse number in parameters: " + text, e);
                }
                switch (index++) {
                    case 0:
                        width = value;
                        break;
                    case 1:
                        precision = value;
                        break;
                }
                expectComma = true;
            }
            if (!finished) {
                throw new IllegalArgumentException("Unexpected error in column type parameters: " + text);
            }
            token = parser.nextToken();
        }
        boolean isUnsigned = false;
        while (token != null) {
            if (!token.isQuotedString() && "unsigned".equalsIgnoreCase(token.getText())) {
                isUnsigned = true;
            }
            token = parser.nextToken();
        }
        this.width = width;
        this.precision = precision;
        this.isUnsigned = isUnsigned;
    }

    /**
     * The underlying data type of the column
     */
    public MySQLDataType getDataType() {
        return dataType;
    }

    /**
     * Usually display width of the number
     */
    public int getWidth() {
        return width;
    }

    /**
     * Usually precision of the number
     */
    public int getPrecision() {
        return precision;
    }

    /**
     * A list of string parameters of the type, e.g. values for enum and set types
     */
    public List<String> getValues() {
        return Collections.unmodifiableList(values);
    }

    /**
     * For sets and enums - value at i position
     */
    public String getValue(int idx) {
        return values.get(idx);
    }

    /**
     * True for unsigned integer types
     */
    public boolean isUnsigned() {
        return isUnsigned;
    }

    @Override
    public String toString() {
        return "MySQLColumnType{" +
                "dataType=" + dataType +
                ", width=" + width +
                ", precision=" + precision +
                ", values=" + values +
                ", isUnsigned=" + isUnsigned +
                '}';
    }

    /**
     * Extract a long value from a number of this type
     */
    private long extractLongFromNumber(Number number) {
        if (isUnsigned()) {
            // Unsigned values need a special mask
            switch (getDataType()) {
                case TINYINT:
                    return number.longValue() & 0xffL;
                case SMALLINT:
                    return number.longValue() & 0xffffL;
                case MEDIUMINT:
                    return number.longValue() & 0xffffffL;
                case INT:
                    return number.longValue() & 0xffffffffL;
            }
        }
        return number.longValue();
    }

    /**
     * Extract a long value from a serialized value of this type
     */
    public long extractLong(Serializable value) {
        if (value instanceof Number) {
            return extractLongFromNumber((Number) value);
        }
        if (value instanceof BitSet) {
            BitSet bs = (BitSet) value;
            long result = 0;
            int index = 0;
            while ((index = bs.nextSetBit(index)) != -1) {
                result |= 1L << index;
                ++index;
            }
            return result;
        }
        if (value instanceof Boolean) {
            return ((Boolean) value) ? 1 : 0;
        }
        throw new IllegalArgumentException("Cannot convert " + value.getClass().getSimpleName() + " to a long");
    }

    /**
     * Extract a double value from a serialized value of this type
     */
    public double extractDouble(Serializable value) {
        if (value instanceof Number) {
            if (isUnsigned()) {
                // Make sure unsigned integers are handled correctly
                return extractLongFromNumber((Number) value);
            }
            return ((Number) value).doubleValue();
        }
        if (value instanceof BitSet) {
            BitSet bs = (BitSet) value;
            double result = 0.0;
            int index = 0;
            while ((index = bs.nextSetBit(index)) != -1) {
                result += Math.pow(2.0, index);
                ++index;
            }
            return result;
        }
        if (value instanceof Boolean) {
            return ((Boolean) value) ? 1.0 : 0.0;
        }
        throw new IllegalArgumentException("Cannot convert " + value.getClass().getSimpleName() + " to a double");
    }

    /**
     * Extract a boolean value from a serialized value of this type
     */
    public boolean extractBoolean(Serializable value) {
        if (value instanceof Number) {
            return ((Number) value).longValue() != 0;
        }
        if (value instanceof BitSet) {
            BitSet bs = (BitSet) value;
            return bs.nextSetBit(0) != -1;
        }
        if (value instanceof Boolean) {
            return (Boolean) value;
        }
        throw new IllegalArgumentException("Cannot convert " + value.getClass().getSimpleName() + " to a boolean");
    }

    /**
     * Extract a string value from a number of this type
     */
    private String extractStringFromNumber(Number number) {
        if (isUnsigned()) {
            // Unsigned values need a special mask
            switch (getDataType()) {
                case TINYINT:
                    return UnsignedLongs.toString(number.longValue() & 0xffL);
                case SMALLINT:
                    return UnsignedLongs.toString(number.longValue() & 0xffffL);
                case MEDIUMINT:
                    return UnsignedLongs.toString(number.longValue() & 0xffffffL);
                case INT:
                    return UnsignedLongs.toString(number.longValue() & 0xffffffffL);
                case BIGINT:
                    return UnsignedLongs.toString(number.longValue());
            }
        }
        return number.toString();
    }

    /**
     * Extract a LocalDate value from a serialized value of this type
     */
    public LocalDate extractLocalDate(Serializable value) {
        return extractLocalDateTime(value).toLocalDate();
    }

    /**
     * Extract a LocalDateTime value from a serialized value of this type
     */
    public LocalDateTime extractLocalDateTime(Serializable value) {
        switch (getDataType()) {
            case DATE:
            case TIMESTAMP:
                if (value instanceof Timestamp) {
                    Timestamp timestamp = (Timestamp) value;
                    return timestamp.toInstant().atZone(DateTimeUtils.MSK).toLocalDateTime();
                }
            case DATETIME:
                if (value instanceof java.sql.Date) {
                    LocalDate date = ((java.sql.Date) value).toLocalDate();
                    return date.atStartOfDay().atZone(DateTimeUtils.MSK).toLocalDateTime();
                } else if (value instanceof Date) {
                    Date date = (Date) value;
                    return date.toInstant().atZone(DateTimeUtils.MSK).toLocalDateTime();
                }
        }
        throw new IllegalArgumentException("Cannot convert " +
                value.getClass().getSimpleName() + " to a LocalDateTime");
    }

    /**
     * Extract a string value from a serialized value of this type
     */
    public String extractString(Serializable value) {
        switch (getDataType()) {
            case BIT:
            case TINYINT:
            case SMALLINT:
            case MEDIUMINT:
            case INT:
            case BIGINT:
            case DECIMAL:
            case FLOAT:
            case DOUBLE:
            case YEAR:
                // Data should be some kind of number
                if (value instanceof Number) {
                    return extractStringFromNumber((Number) value);
                }
                return value.toString();
            case DATE:
            case TIMESTAMP:
                if (value instanceof Timestamp) {
                    Timestamp timestamp = (Timestamp) value;
                    return DATE_TIME_FORMATTER.format(timestamp.toInstant());
                }
                return value.toString();
            case TIME:
                // Data should be some kind of java.sql.Date/Time
                return value.toString();
            // костылик для дефекта https://github.com/shyiko/mysql-binlog-connector-java/issues/193
            case DATETIME:
                // парсер десериализует в java.util.Date только значения колонок типа datetime (см.
                // AbstractRowsEventDataDeserializer:175)
                if (value instanceof Date) {
                    Date date = (Date) value;
                    // Проблема с таймзоной растет из метода AbstractRowsEventDataDeserializer::deserializeDatetimeV2,
                    // который парсит значение поля типа datetime без учета таймзоны (в UTC),
                    // из-за чего полученная в результате временная метка имеет разницу с оригиналом в размере
                    // смещения московского часового пояса (+3:00)
                    return UTC_DATE_TIME_FORMATTER.format(date.toInstant());
                }
                return value.toString();
            case CHAR:
            case VARCHAR:
            case TINYTEXT:
            case TEXT:
            case MEDIUMTEXT:
            case LONGTEXT:
                if (value instanceof byte[]) {
                    return new String((byte[]) value, StandardCharsets.UTF_8);
                }
                if (value instanceof String) {
                    return (String) value;
                }
                throw new IllegalArgumentException(
                        "Argument type " + value.getClass() + " does not match column type " + getDataType());
            case BINARY:
            case VARBINARY:
            case TINYBLOB:
            case BLOB:
            case MEDIUMBLOB:
            case LONGBLOB:
                if (value instanceof byte[]) {
                    return Base64.getEncoder().encodeToString((byte[]) value);
                }
                if (value instanceof String) {
                    return (String) value;
                }
                throw new IllegalArgumentException(
                        "Argument type " + value.getClass() + " does not match column type " + getDataType());
            case ENUM: {
                if (value instanceof Number) {
                    int index = ((Number) value).intValue();
                    if (index <= 0) {
                        return "";
                    }
                    --index;
                    if (index < values.size()) {
                        return values.get(index);
                    }
                }
                return value.toString();
            }
            case SET: {
                if (value instanceof Number) {
                    long bits = ((Number) value).longValue();
                    StringBuilder sb = new StringBuilder();
                    for (int i = 0; i < values.size(); ++i) {
                        if ((bits & (1L << i)) != 0) {
                            if (sb.length() != 0) {
                                sb.append(',');
                            }
                            sb.append(values.get(i));
                        }
                    }
                    return sb.toString();
                }
                return value.toString();
            }
            case JSON:
                if (value instanceof byte[]) {
                    byte[] byteValue = (byte[]) value;
                    if (byteValue.length == 0) {
                        return "";
                    }
                    try {
                        // Сначала декодируем из бинарного представления
                        String json = JsonBinary.parseAsString(byteValue);

                        // А потом делаем pretty-print, соответствующий тому, который использует mysql
                        // Одновременно с этим применяем workaround для бага https://bugs.mysql.com/bug.php?id=88230
                        // На будущее: если ошибок в обработке не будет, то можно будет убрать fallback'и в catch'ах
                        try {
                            return Mysql57CompatibleJsonFormatter.format(json);
                        } catch (Mysql57CompatibleJsonFormatter.Mysql57CompatibleJsonFormatException e) {
                            logger.error(String.format("Exception while trying to format json '%s'", json), e);

                            // Если не вышло применить workaround, то хотя бы просто отформатируем так же, как MySQL
                            try {
                                JsonNode jsonNode = OBJECT_MAPPER.readTree(json);
                                return OBJECT_MAPPER.writeValueAsString(jsonNode);
                            } catch (IOException exc) {
                                // Если и так не получилось, тогда возвращаем исходный json
                                logger.error(
                                        String.format("Exception while trying to pretty-print json '%s'", json), exc);
                                return json;
                            }
                        }
                    } catch (IOException ex) {
                        throw new IllegalArgumentException("Can't format json as string", ex);
                    }
                }
                return value.toString();
            default:
                // By default simply call the toString() method on the value
                return value.toString();
        }
    }

    /**
     * Parses the specified string value into an expected Serializable value
     */
    public Serializable parseValue(String value) {
        if (value == null) {
            // Значение null возвращается как есть
            return null;
        }
        switch (getDataType()) {
            case BIT:
                if (value.isEmpty()) {
                    return 0;
                }
                if (value.startsWith("b'") && value.endsWith("'")) {
                    long result = 0;
                    for (int index = 2; index < value.length() - 1; ++index) {
                        char c = value.charAt(index);
                        if (c == '0') {
                            result <<= 1;
                        } else if (c == '1') {
                            result <<= 1;
                            result |= 1;
                        } else {
                            throw new IllegalArgumentException("Cannot parse " + value + " as mysql bit");
                        }
                    }
                    return result;
                }
                return UnsignedLongs.parseUnsignedLong(value);
            case TINYINT:
                if (value.isEmpty()) {
                    return (byte) 0;
                }
                if (isUnsigned) {
                    return (int) UnsignedBytes.parseUnsignedByte(value);
                } else {
                    return (int) Byte.parseByte(value);
                }
            case SMALLINT:
                if (value.isEmpty()) {
                    return (short) 0;
                }
                if (isUnsigned) {
                    return UnsignedInts.parseUnsignedInt(value);
                } else {
                    return (int) Short.parseShort(value);
                }
            case MEDIUMINT:
            case INT:
            case YEAR:
                if (value.isEmpty()) {
                    return 0;
                }
                if (isUnsigned) {
                    return UnsignedInts.parseUnsignedInt(value);
                } else {
                    return Integer.parseInt(value);
                }
            case BIGINT:
                if (value.isEmpty()) {
                    return (long) 0;
                }
                if (isUnsigned) {
                    return UnsignedLongs.parseUnsignedLong(value);
                } else {
                    return Long.parseLong(value);
                }
            case DECIMAL:
                if (value.isEmpty()) {
                    return BigDecimal.valueOf(0);
                }
                return new BigDecimal(value);
            case FLOAT:
                if (value.isEmpty()) {
                    return 0.0f;
                }
                return Float.parseFloat(value);
            case DOUBLE:
                if (value.isEmpty()) {
                    return 0.0;
                }
                return Double.parseDouble(value);
            case DATE:
            case DATETIME:
            case TIMESTAMP:
                if (value.startsWith("0000-00-00") || value.equals("CURRENT_TIMESTAMP")) {
                    return null;
                }
                // FIXME: возможно стоит сделать нормальный парсер
                throw new UnsupportedOperationException("Cannot parse " + getDataType() + " value '" + value + "'");
            case TIME:
                if (value.startsWith("00:00:00")) {
                    return null;
                }
                // FIXME: возможно стоит сделать нормальный парсер
                throw new UnsupportedOperationException("Cannot parse " + getDataType() + " value '" + value + "'");
            case CHAR:
            case VARCHAR:
            case TINYTEXT:
            case TEXT:
            case MEDIUMTEXT:
            case LONGTEXT:
                return value;
            case BINARY:
            case VARBINARY:
            case TINYBLOB:
            case BLOB:
            case MEDIUMBLOB:
            case LONGBLOB:
                // FIXME: возможно стоит сделать нормальный парсер
                throw new UnsupportedOperationException("Cannot parse " + getDataType() + " value '" + value + "'");
            case ENUM: {
                int index = values.indexOf(value);
                if (index != -1) {
                    return index + 1;
                }
                return 0;
            }
            case SET: {
                long bits = 0;
                for (String part : value.split(",")) {
                    part = part.trim();
                    int index = values.indexOf(part);
                    if (index != -1) {
                        bits |= 1 << index;
                    }
                }
                return bits;
            }
        }
        throw new UnsupportedOperationException("Cannot parse values of type " + getDataType());
    }

    private Serializable parseDefaultValueForCaching(String defaultValue) {
        Serializable value = parseValue(defaultValue);
        return value != null ? value : NullValue.INSTANCE;
    }

    public Serializable getCachedDefaultValue(String defaultValue) {
        if (defaultValue == null) {
            return null;
        }
        Serializable result = cachedDefaultValues.get(defaultValue);
        if (result == null) {
            // Вызов computeIfAbsent дороже, чем простой get, вызываем его только при необходимости
            result = cachedDefaultValues.computeIfAbsent(defaultValue, this::parseDefaultValueForCaching);
        }
        return result != NullValue.INSTANCE ? result : null;
    }

    public static MySQLColumnType getCached(String columnType) {
        MySQLColumnType result = cachedColumnTypes.get(columnType);
        if (result == null) {
            // Вызов computeIfAbsent дороже, чем простой get, вызываем его только при необходимости
            result = cachedColumnTypes.computeIfAbsent(columnType, MySQLColumnType::new);
        }
        return result;
    }

    private static class Token {
        private String text;
        private boolean isQuotedString;

        public Token(char ch) {
            this(Character.toString(ch));
        }

        public Token(String text) {
            this(text, false);
        }

        public Token(String text, boolean isQuotedString) {
            this.text = text;
            this.isQuotedString = isQuotedString;
        }

        public String getText() {
            return text;
        }

        public boolean isQuotedString() {
            return isQuotedString;
        }
    }

    private static class Parser {
        private String text;
        private int index;
        private boolean insideParenthesis;

        public Parser(String text) {
            this.text = text;
            index = 0;
        }

        private int skipWhitespace() {
            while (index < text.length() && Character.isWhitespace(text.charAt(index))) {
                ++index;
            }
            return index;
        }

        private String readLiteral(char quote) {
            StringBuilder sb = new StringBuilder(text.length() - index);
            while (index < text.length()) {
                char ch = text.charAt(index++);
                if (ch == quote) {
                    if (index < text.length() && text.charAt(index) == quote) {
                        // an escaped quote character
                        sb.append(ch);
                        ++index;
                        continue;
                    }
                    break;
                }
                if (ch == '\\' && index < text.length()) {
                    ch = text.charAt(index++);
                    switch (ch) {
                        case '0':
                            ch = '\0';
                            break;
                        case 'b':
                            ch = '\b';
                            break;
                        case 'n':
                            ch = '\n';
                            break;
                        case 'r':
                            ch = '\r';
                            break;
                        case 't':
                            ch = '\t';
                            break;
                        case 'Z':
                            ch = '\032'; // ASCII 26 (32 in octal)
                            break;
                    }
                }
                sb.append(ch);
            }
            return sb.toString();
        }

        public Token nextToken() {
            int start = skipWhitespace();
            if (index >= text.length()) {
                // no tokens left
                return null;
            }
            char ch = text.charAt(index++);
            switch (ch) {
                case '(':
                    return new Token(ch);
                case ')':
                    return new Token(ch);
                case ',':
                    return new Token(ch);
                case '\'':
                    return new Token(readLiteral(ch), true);
                case '"':
                    return new Token(readLiteral(ch), true);
            }
            while (index < text.length()) {
                ch = text.charAt(index);
                if (ch == '(' || ch == ')' || ch == ',' || ch == '\'' || ch == '"' || Character.isWhitespace(ch)) {
                    break;
                }
                ++index;
            }
            return new Token(text.substring(start, index));
        }
    }

    /**
     * Синглтон для кеширования null значений
     */
    private static final class NullValue implements Serializable {
        static final NullValue INSTANCE = new NullValue();

        @Override
        public String toString() {
            return "null";
        }
    }
}
