package ru.yandex.direct.mysql;

import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.mysql.cj.jdbc.MysqlDataSource;
import com.mysql.cj.jdbc.exceptions.CommunicationsException;

import ru.yandex.direct.process.Docker;
import ru.yandex.direct.process.DockerContainer;
import ru.yandex.direct.process.DockerRunner;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.HostPort;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.ThreadUtils;

public class MySQLDockerContainer extends DockerContainer implements MySQLInstance {
    // кастомный образ, строится в /direct/pachages/docker/yandex-direct-percona
    public static final String DEFAULT_IMAGE = "registry.yandex.net/direct/yandex-direct-percona:5.7.34-2";

    private HostPort hostPort;

    public MySQLDockerContainer(Docker docker) throws InterruptedException {
        this(createRunner(docker));
    }

    public MySQLDockerContainer(DockerRunner runner) throws InterruptedException {
        super(runner);
        boolean success = false;
        try {
            hostPort = getPublishedPort(3306);
            success = true;
        } finally {
            if (!success) {
                close();
            }
        }
    }

    public static DockerRunner createRunner(Docker docker) {
        return createRunner(docker, DEFAULT_IMAGE);
    }

    public static DockerRunner createRunner(Docker docker, String image) {
        return new DockerRunner(docker, image)
                .withEnvironment("MYSQL_ALLOW_EMPTY_PASSWORD", "1")
                // 3307, чтобы не было конфликтов в случае, если запущен обычный mysql на компьютере
                .withPublishedPort(3307, 3306);
    }

    public HostPort getHostPort() {
        return hostPort;
    }

    @Override
    public String getHost() {
        return getHostPort().getHost();
    }

    @Override
    public int getPort() {
        return getHostPort().getPort();
    }

    @Override
    public String getUsername() {
        return "root";
    }

    @Override
    public String getPassword() {
        return "";
    }

    @Override
    public void start() throws InterruptedException {
        super.start();
        hostPort = getPublishedPort(3306);
    }

    @Override
    public Connection connect() throws SQLException {
        return connect(Duration.ofSeconds(20));
    }

    public Connection connect(Duration timeout) throws SQLException {
        return connect(timeout, null);
    }

    public Connection connect(Duration timeout, @Nullable Consumer<MysqlDataSource> configurer) throws SQLException {
        MonotonicTime deadline = NanoTimeClock.now().plus(timeout);
        while (true) {
            try {
                return MySQLUtils.connect(getHost(), getPort(), getUsername(), getPassword(), null, configurer);
            } catch (CommunicationsException exc) {
                Duration remainingTimeout = deadline.minus(NanoTimeClock.now());

                if (remainingTimeout.isNegative()) {
                    throw new IllegalStateException("Failed to connect to mysql container", exc);
                }
                ThreadUtils.sleepUninterruptibly(CommonUtils.min(Duration.ofMillis(500), remainingTimeout));

                checkExitCode(CommonUtils.max(Duration.ofSeconds(5), remainingTimeout));
            }
        }
    }

    @Override
    public void awaitConnectivity(Duration timeout) {
        try {
            connect(timeout).close();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }
}
