package ru.yandex.direct.mysql;

import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.process.ProcessCommunicator;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.Interrupts;

public class MySQLServer implements MySQLInstance, AutoCloseable {
    public static final ImmutableList<String> MYSQL_SERVER_PARAMS = ImmutableList.of(
            "--character-set-server=utf8",
            "--skip-name-resolve"
    );
    public static final ImmutableList<String> MYSQL_OPTIMIZATION_PARAMS = ImmutableList.of(
            "--innodb-flush-method=nosync",
            "--innodb-buffer-pool-dump-at-shutdown=OFF",
            "--innodb-buffer-pool-load-at-startup=OFF",
            "--innodb_log_file_size=1M",
            "--innodb-file-per-table=OFF",
            "--innodb-stats-persistent=OFF",
            "--innodb-doublewrite=0",
            "--innodb-flush-log-at-trx-commit=0",
            "--innodb-monitor-disable=all",

            "--performance-schema=OFF",
            "--sync-frm=OFF",

            "--skip-ssl"
    );

    private static final Logger logger = LoggerFactory.getLogger(MySQLServer.class);
    private static final AtomicInteger counter = new AtomicInteger(0);

    private static final Duration MAX_CONNECT_DELAY = Duration.ofMillis(100);
    private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(15);

    private final String host;
    private final int port;
    private final ProcessCommunicator processCommunicator;

    public MySQLServer(String host, int port, ProcessBuilder processBuilder, Duration gracefulStopTimeout) {
        this.host = host;
        this.port = port;
        int index = counter.incrementAndGet();
        this.processCommunicator = new ProcessCommunicator.Builder(processBuilder)
                .withThreadNamePrefix("mysql-server-" + index)
                .withGracefulStopTimeout(gracefulStopTimeout)
                .withStdoutReader(new ProcessCommunicator.LineReader(StandardCharsets.UTF_8, logger::debug))
                .withStderrReader(new ProcessCommunicator.LineReader(StandardCharsets.UTF_8, logger::error))
                .build();
    }

    /**
     * Attempts to connect the the database
     */
    public Connection connect(String user, String password, long timeout, TimeUnit unit) throws SQLException {
        boolean exited = false;
        long deadline = System.nanoTime() + unit.toNanos(timeout);
        while (!exited) {
            try {
                return MySQLUtils.connect(getHost(), getPort(), user, password);
            } catch (SQLRecoverableException e) {
                long remainingTimeout = deadline - System.nanoTime();
                if (remainingTimeout <= 0) {
                    throw e;
                }
                exited = Interrupts.waitUninterruptibly(
                        Duration.ofNanos(Math.min(remainingTimeout, MAX_CONNECT_DELAY.toNanos())),
                        processCommunicator::waitFor
                );
            }
        }
        throw new MySQLServerException("Server stopped before a connect was successful");
    }

    /**
     * Attempts to connect to the database using the default login and password
     */
    public Connection connect(long timeout, TimeUnit unit) throws SQLException {
        return connect(getUsername(), getPassword(), timeout, unit);
    }

    /**
     * Attempts to connect to the database using the default login and password
     */
    @Override
    public Connection connect() throws SQLException {
        return connect(DEFAULT_CONNECT_TIMEOUT);
    }

    @Override
    public Connection connect(Duration timeout) throws SQLException {
        return connect(getUsername(), getPassword(), timeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public String getHost() {
        return host;
    }

    @Override
    public int getPort() {
        return port;
    }

    @Override
    public String getUsername() {
        return "root";
    }

    @Override
    public String getPassword() {
        return "";
    }

    public boolean await(long timeout, TimeUnit unit) throws InterruptedException {
        return await(Duration.of(timeout, CommonUtils.chronoUnit(unit)));
    }

    public boolean await(Duration timeout) throws InterruptedException {
        return processCommunicator.waitFor(timeout);
    }

    public void await() throws InterruptedException {
        processCommunicator.waitFor();
    }

    @Override
    public void close() {
        processCommunicator.close();
    }

    @Override
    public void awaitConnectivity(Duration timeout) {
        try {
            connect(timeout.toMillis(), TimeUnit.MILLISECONDS).close();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }
}
