package ru.yandex.direct.mysql;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.github.zafarkhaja.semver.Version;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import ru.yandex.direct.process.ProcessExitStatusException;
import ru.yandex.direct.process.ProcessForciblyDestroyedException;
import ru.yandex.direct.process.Processes;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Class for managing a temporary mysqld process
 * <p>
 * Example usage:
 * <pre>
 *     MySQLServerBuilder server = new MySQLServerBuilder();
 *     server.setDataAndConfigDir(Files.createTempDirectory("mysql"));
 *     try {
 *         server.initializeDataDir();
 *         try(MySQLServer instance = server.start()) {;
 *             try (Connection conn = instance.connect()) {
 *                 // use the connection
 *             }
 *         }
 *     } finally {
 *         MySQLUtils.deleteRecursive(server.getDataDir());
 *     }
 * </pre>
 */
public class MySQLServerBuilder {
    /**
     * DIRECT-81042
     * <p>
     * В силу того, что mysqld нельзя запустить на случайном порту, приходится сначала брать случайный порт,
     * потом освобождать его, а потом запускать mysqld на этом порту. В течение нескольки секунд этот порт свободен.
     * <p>
     * Готовность mysqld принимать запросы реализуется периодическим опросом порта.
     * <p>
     * Если запускать временные mysqld на случайном порту из /proc/sys/net/ipv4/ip_local_port_range, появляется
     * состояние гонки, что в редких случаях может привести к созданию tcp-соединения с совпадающими
     * (src_host, src_port) и (dst_host, dst_port). mysql-connector не готов к такому сюрпризу, он пытается бесконечно
     * читать из него, в итоге приложение зависает.
     * <p>
     * Быстрое и грязное решение: пытаться поднять mysqld не на 127.0.0.1, а на каком-то другом адресе,
     * тогда даже если src_port и dst_port совпадут, src_host не будет совпадать с dst_host.
     * <p>
     * При этом не все операционные системы позволяют такое вытворять. Linux, который используется в продакшне,
     * позволяет. MacOS, которая может запускать этот код только внутри функциональных тестов, не позволяет.
     */
    @SuppressWarnings("squid:S1313")
    private static final List<String> LOOPBACK_ADDRESSES = ImmutableList.of(
            "127.210.213.90",    // Chosen once random loopback IP address
            "127.0.0.1");

    private List<String> mysqldBinary;
    private Path mysqlInstallDbBinary;
    private int port;
    private int serverId;
    private Path dataDir;
    private Path socketFile;
    private List<String> extraArgs;
    private Path errorLog;
    private Duration gracefulStopTimeout;
    private Map<String, String> environment;
    private boolean skipNetworking;
    private Path config;
    private Path workDir;
    private boolean noSync;
    private boolean needStrace = false;
    private boolean aioDisabled = false;
    private Function<String, TempDirectory> tempDirectoryProvider;

    public MySQLServerBuilder(List<String> mysqldBinary, Path mysqlInstallDbBinary, int port, int serverId,
                              Path dataDir, Path socketFile,
                              List<String> extraArgs, Path errorLog, Duration gracefulStopTimeout,
                              Map<String, String> environment, boolean skipNetworking, Path config, Path workDir,
                              Function<String, TempDirectory> tempDirectoryProvider) {
        this.mysqldBinary = mysqldBinary;
        this.mysqlInstallDbBinary = mysqlInstallDbBinary;
        this.port = port;
        this.serverId = serverId;
        this.dataDir = dataDir;
        this.socketFile = socketFile;
        this.extraArgs = extraArgs;
        this.errorLog = errorLog;
        this.gracefulStopTimeout = gracefulStopTimeout;
        this.environment = environment;
        this.skipNetworking = skipNetworking;
        this.config = config;
        this.workDir = workDir;
        this.tempDirectoryProvider = tempDirectoryProvider;
    }

    public MySQLServerBuilder() {
        this(
                Arrays.asList(Paths.get("mysqld").toString()),
                Paths.get("mysql_install_db"),
                0,
                0,
                null,
                null,
                new ArrayList<>(),
                null,
                Duration.ofSeconds(30),
                new HashMap<>(),
                false,
                null,
                null,
                null
        );
    }

    public MySQLServerBuilder copy() {
        return new MySQLServerBuilder(
                mysqldBinary,
                mysqlInstallDbBinary,
                port,
                serverId,
                dataDir,
                socketFile,
                new ArrayList<>(extraArgs),
                errorLog,
                gracefulStopTimeout,
                environment,
                skipNetworking,
                config,
                workDir,
                tempDirectoryProvider
        );
    }

    public int getPort() {
        return port;
    }

    public int getServerId() {
        return serverId;
    }

    public Path getDataDir() {
        return dataDir;
    }

    public Path getConfig() {
        return config;
    }

    public Path getSocketFile() {
        return socketFile;
    }

    public List<String> getExtraArgs() {
        return extraArgs;
    }

    public MySQLServerBuilder setMysqldBinary(Path mysqldBinary) {
        return setMysqldBinary(Collections.singletonList(mysqldBinary.toString()));
    }

    public MySQLServerBuilder setMysqldBinary(List<String> mysqldBinary) {
        this.mysqldBinary = mysqldBinary;
        return this;
    }

    public MySQLServerBuilder setMysqlInstallDbBinary(Path mysqlInstallDbBinary) {
        this.mysqlInstallDbBinary = mysqlInstallDbBinary;
        return this;
    }

    public MySQLServerBuilder setPort(int port) {
        this.port = port;
        return this;
    }

    public MySQLServerBuilder setServerId(int serverId) {
        this.serverId = serverId;
        return this;
    }

    public boolean isNoSync() {
        return noSync;
    }

    public MySQLServerBuilder withNoSync(boolean noSync) {
        this.noSync = noSync;
        return this;
    }

    public MySQLServerBuilder setDataAndConfigDir(Path dataAndConfigDir) {
        setDataDir(dataAndConfigDir.resolve("data"));
        setConfig(dataAndConfigDir.resolve("mysqld.conf"));

        if (!getConfig().toFile().exists()) {
            try {
                Files.write(getConfig(), MySQLServerDefaultConfig.getData());
            } catch (IOException exc) {
                throw new Checked.CheckedException(exc);
            }
        }

        return this;
    }

    public MySQLServerBuilder setDataDir(Path dataDir) {
        if (this.dataDir != null) {
            throw new IllegalStateException("dataDir is already set to: " + this.dataDir);
        }
        this.dataDir = dataDir;
        return this;
    }

    public MySQLServerBuilder setConfig(Path config) {
        if (this.config != null) {
            throw new IllegalStateException("config is already set to: " + this.config);
        }
        this.config = config;
        return this;
    }

    public MySQLServerBuilder setSocketFile(Path socketFile) {
        this.socketFile = socketFile;
        return this;
    }

    public MySQLServerBuilder setWorkDir(Path workDir) {
        this.workDir = workDir;
        return this;
    }

    public MySQLServerBuilder addExtraArgs(List<String> extraArgs) {
        this.extraArgs.addAll(extraArgs);
        return this;
    }

    public MySQLServerBuilder addExtraArgs(String... extraArgs) {
        this.extraArgs.addAll(Arrays.asList(extraArgs));
        return this;
    }

    public MySQLServerBuilder setErrorLog(Path errorLog) {
        this.errorLog = errorLog;
        return this;
    }

    public MySQLServerBuilder setGracefulStopTimeout(Duration gracefulStopTimeout) {
        this.gracefulStopTimeout = gracefulStopTimeout;
        return this;
    }

    public MySQLServerBuilder addEnvironment(String key, String value) {
        environment.put(key, value);
        return this;
    }

    public MySQLServerBuilder setSkipNetworking(boolean skipNetworking) {
        this.skipNetworking = skipNetworking;
        return this;
    }

    public MySQLServerBuilder withTempDirectoryProvider(Function<String, TempDirectory> tempDirectoryProvider) {
        this.tempDirectoryProvider = tempDirectoryProvider;
        return this;
    }

    public MySQLServerBuilder withNeedStrace(boolean needStrace) {
        this.needStrace = needStrace;
        return this;
    }

    public MySQLServerBuilder disableAio() {
        this.aioDisabled = true;
        return this;
    }

    public TempDirectory createTempDirectory(String name) {
        return tempDirectoryProvider != null
                ? tempDirectoryProvider.apply(name)
                : new TempDirectory("mysql-" + name);
    }

    private void addErrorLogArgs(List<String> args) {
        if (errorLog == null) {
            args.add("--log-error=stderr");
        } else {
            if (errorLog.getNameCount() == 1) {
                Path dir = Preconditions.checkNotNull(getDataDir(), "Data directory not specified");
                args.add("--log-error=" + dir.resolve(errorLog));
            } else {
                args.add("--log-error=" + errorLog);
            }
        }
    }

    public void initializeDataDir(boolean useMysqlInstallDb) throws InterruptedException {
        Preconditions.checkNotNull(getDataDir(), "Data directory not specified");
        Preconditions.checkNotNull(getConfig(), "Config path not specified");

        MySQLServerBuilder initializer = this.copy().setSkipNetworking(true);
        if (useMysqlInstallDb) {
            initializer
                    .setServerId(0)
                    .setMysqldBinary(mysqlInstallDbBinary)
                    .addExtraArgs("--force");
        } else {
            initializer.addExtraArgs("--initialize-insecure=on");
        }

        Processes.checkCall(initializer.createProcessBuilder());
    }

    public void initializeDataDir() throws InterruptedException {
        initializeDataDir(requiresMysqlInstallDb(getServerVersion()));
    }

    public MySQLServer start() {
        InetSocketAddress addr = getHostPort(port);
        String host = addr.getAddress().getHostAddress();
        int port = addr.getPort();
        return new MySQLServer(host, port, createProcessBuilder(addr), gracefulStopTimeout);
    }

    private static InetSocketAddress getHostPort(int port) {
        IOException ioException = null;
        for (String loopbackAddress : LOOPBACK_ADDRESSES) {
            try (ServerSocket socket = new ServerSocket(port, 128, InetAddress.getByName(loopbackAddress))) {
                return new InetSocketAddress(loopbackAddress, socket.getLocalPort());
            } catch (IOException e) {
                ioException = e;
            }
        }
        Objects.requireNonNull(ioException);
        throw new MySQLServerException("Server port allocation failed", ioException);
    }

    private ProcessBuilder createProcessBuilder() {
        return createProcessBuilder(getHostPort(0));
    }

    private ProcessBuilder createProcessBuilder(InetSocketAddress listenAddr) {
        List<String> args = new ArrayList<>();

        if (needStrace) {
            args.addAll(List.of("strace", "-s100", "-ttT", "-f"));
        }
        args.addAll(mysqldBinary);

        String host = listenAddr.getAddress().getHostAddress();
        int port = listenAddr.getPort();

        if (getConfig() == null) {
            args.add("--no-defaults");
        } else {
            args.add("--defaults-file=" + getConfig().toAbsolutePath());
        }

        args.addAll(getExtraArgs()); // Есть опции, которые можно указывать только самыми первыми,
        // поэтому сначала extra, а потом все остальное
        if (dataDir != null) {
            args.add("--datadir=" + dataDir.toAbsolutePath());
        }

        if (skipNetworking) {
            args.add("--skip-networking");
        } else {
            args.add("--socket=" + (getSocketFile() != null ? getSocketFile().toAbsolutePath() : ""));
            args.add("--bind-address=" + host);
            args.add("--port=" + (port != 0 ? port : ""));
        }
        if (getServerId() != 0) {
            args.add("--server-id=" + getServerId());
        }
        addErrorLogArgs(args);
        args.add("--secure-file-priv=");

        if (noSync) {
            // DIRECT-79652 было обнаружено, что без этих опции создание таблиц может работать десятки секунд.
            args.add("--skip-sync-frm");
            args.add("--sync-binlog=0");
            args.add("--sync-master-info=0");
            args.add("--sync-relay-log=0");
            args.add("--sync-relay-log-info=0");
        }

        if (aioDisabled) {
            args.add("--innodb_use_native_aio=0");
        }

        ProcessBuilder builder = new ProcessBuilder(args);
        builder.environment().putAll(environment);

        if (workDir != null) {
            builder.directory(workDir.toFile());
        }

        return builder;
    }

    public Version getServerVersion() throws InterruptedException {
        String output = Processes.checkOutput(this.copy().addExtraArgs("--version").createProcessBuilder());
        Matcher matcher = Pattern.compile("\\sVer\\s+(.+?)[\\s\\-]").matcher(output);
        if (!matcher.find()) {
            throw new IllegalStateException("Can't parse version: " + output);
        } else {
            return Version.valueOf(matcher.group(1));
        }
    }

    public static boolean requiresMysqlInstallDb(Version version) {
        /*
        > mysql_install_db is deprecated as of MySQL 5.7.6 because its functionality has been integrated
        > into mysqld, the MySQL server.

        http://dev.mysql.com/doc/refman/5.7/en/mysql-install-db.html
        */
        return version.lessThan(Version.forIntegers(5, 7, 6));
    }

    @SuppressWarnings("squid:S1166") // умышленно не перевыбрасываем пойманные исключения
    public boolean mysqldIsAvailable() throws InterruptedException {
        try {
            getServerVersion();
            return true;
        } catch (ProcessExitStatusException | ProcessForciblyDestroyedException exc) {
            return false;
        } catch (Checked.CheckedException exc) {
            if (exc.unwrapped() instanceof IOException) {
                return false;
            } else {
                throw exc;
            }
        }
    }
}
