package ru.yandex.direct.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.github.shyiko.mysql.binlog.event.Event;
import com.github.shyiko.mysql.binlog.event.EventHeaderV4;
import com.github.shyiko.mysql.binlog.event.EventType;
import com.github.shyiko.mysql.binlog.event.QueryEventData;
import com.google.common.primitives.Ints;
import com.mysql.cj.jdbc.MysqlDataSource;

import ru.yandex.direct.utils.MySQLQuote;

import static ru.yandex.direct.utils.DateTimeUtils.MOSCOW_TIMEZONE;

public class MySQLUtils {
    // флаги binlog-событий
    public static final int LOG_EVENT_THREAD_SPECIFIC_F = 4;
    public static final int LOG_EVENT_SUPPRESS_USE_F = 8;

    /**
     * Квотирует идентификатор как это делает mysqldump, для использования напрямую в sql.
     */
    public static String quoteName(String name) {
        return MySQLQuote.quoteName(name);
    }

    /**
     * Квотирует идентификатор как это делает mysqldump, для использования напрямую в sql в качестве выражения для like.
     */
    public static String quoteLike(String name) {
        return MySQLQuote.quoteLike(name);
    }

    /**
     * Выполняет простой sql запрос.
     */
    public static int executeUpdate(Connection conn, String sql) throws SQLException {
        try (PreparedStatement stmt = conn.prepareStatement(sql)) {
            return stmt.executeUpdate();
        }
    }

    /**
     * Блокирует запись во все таблицы, например для получения консистентного снепшота данных.
     */
    public static void lockTablesForRead(Connection conn) throws SQLException {
        // mysqldump использует это для уменьшения риска длительных блокировок на апдейтах
        executeUpdate(conn, "FLUSH LOCAL TABLES");
        // по факту блокирует все таблицы и замораживает изменения данных
        executeUpdate(conn, "FLUSH TABLES WITH READ LOCK");
    }

    /**
     * Разблокирует все залоченные таблицы
     */
    public static void unlockTables(Connection conn) throws SQLException {
        executeUpdate(conn, "UNLOCK TABLES");
    }

    public static Connection connect(String host, int port, String username, String password) throws SQLException {
        return connect(host, port, username, password, null);
    }

    public static Connection connect(String host, int port,
                                     @Nullable String username, @Nullable String password,
                                     @Nullable Duration timeout) throws SQLException {
        return connect(host, port, username, password, timeout, null);
    }

    public static Connection connect(String host, int port,
                                     @Nullable String username, @Nullable String password,
                                     @Nullable Duration timeout,
                                     @Nullable Consumer<MysqlDataSource> configurer
    ) throws SQLException {
        /*
        Каким-то образом clickhouse вмешивается в работу DriverManager, пытается распарсить этот мускульный УРЛ
        и падает с исключением. Поэтому перешел на использование MysqlDataSource.

        return DriverManager.getConnection(
            String.format("jdbc:mysql://address=(protocol=tcp)(host=%s)(port=%d)", host, port),
            username,
            password
        );
        */
        MysqlDataSource source = new MysqlDataSource();
        source.setServerName(host);
        source.setPort(port);
        source.setUser(username);
        source.setPassword(password);
        source.setZeroDateTimeBehavior("CONVERT_TO_NULL");
        source.setServerTimezone(MOSCOW_TIMEZONE);
        if (timeout != null) {
            source.setConnectTimeout(Ints.saturatedCast(timeout.toMillis()));
        }
        source.setUseCompression(true);
        if (configurer != null) {
            configurer.accept(source);
        }
        return source.getConnection();
    }

    public static boolean looksLikeThreadSpecific(Event event) {
        EventHeaderV4 header = event.getHeader();
        if ((header.getFlags() & LOG_EVENT_THREAD_SPECIFIC_F) == 0) {
            return false;
        }

        if (header.getEventType() == EventType.QUERY) {
            var data = (QueryEventData) event.getData();
            var tokenizer = new SqlWordsParser(data.getSql());
            if (tokenizer.nextWord().equalsIgnoreCase("DROP")
                    && tokenizer.nextWord().equalsIgnoreCase("TABLE")
            ) {
                // DROP TABLE ошибочно помечаются флагом thread-specifiс, для них возвращаем false
                // при этом DROP TEMPORARY TABLE останется thread-specific
                // https://bugs.mysql.com/bug.php?id=104451
                return false;
            }
        }

        return true;
    }

    public static String extractFirstWordFromSQL(String query) {
        var tokenizer = new SqlWordsParser(query);
        return tokenizer.nextWord();
    }

    static class SqlWordsParser {
        final String query;
        int pos;
        final int end;

        SqlWordsParser(String query) {
            this.query = query == null ? "" : query;
            this.pos = 0;
            this.end = this.query.length();
        }

        @Nonnull
        String nextWord() {
            // first skip all whitespaces and comments
            while (pos < end) {
                char c = query.charAt(pos++);
                if (c == '-' && pos < end && query.charAt(pos) == '-') {
                    // a single-line comment, skip until the next line
                    ++pos;
                    while (pos < end) {
                        char cc = query.charAt(pos++);
                        if (cc == '\n') {
                            break;
                        }
                    }
                    continue;
                }
                if (c == '/' && pos < end && query.charAt(pos) == '*') {
                    // start of multiline comment
                    ++pos;
                    int nesting = 1;
                    while (nesting > 0 && pos < query.length()) {
                        char cc = query.charAt(pos++);
                        if (cc == '*' && pos < end && query.charAt(pos) == '/') {
                            // end of multiline comment
                            --nesting;
                            ++pos;
                        }
                        if (cc == '/' && pos < end && query.charAt(pos) == '*') {
                            // start of another comment
                            ++nesting;
                            ++pos;
                        }
                    }
                    continue;
                }
                if (Character.isWhitespace(c)) {
                    continue;
                }
                if (!Character.isLetter(c)) {
                    break;
                }
                int start = pos - 1;
                while (pos < end && Character.isLetterOrDigit(query.charAt(pos))) {
                    ++pos;
                }
                return query.substring(start, pos);
            }
            // end of string or weird non-whitespace character found
            return "";
        }
    }

    public static void loosenRestrictions(Connection schemaConnection) throws SQLException {
        executeUpdate(schemaConnection, "SET SQL_NOTES=0");
        executeUpdate(schemaConnection, "SET UNIQUE_CHECKS=0");
        executeUpdate(schemaConnection, "SET FOREIGN_KEY_CHECKS=0");
        executeUpdate(schemaConnection, "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO'");
    }

    /**
     * Метод подбирает число, подходящее как server id.
     * <p>
     * DIRECT-78909: У каждого соединения с mysql должен быть свой уникальный идентификатор. Если к одному mysql
     * подключится два binlog-клиента с одинаковым server id, то один из них в итоге будет убит, причём может быть убит
     * не сразу.
     * <p>
     * См. https://github.com/percona/percona-server/blob/5.7/sql/rpl_master.cc#L495-L505
     *
     * @param initialServerId Исходный server id или null
     * @returns initialServerId, если он не null, иначе случайное число, подходящее как mysql server id.
     */
    public static int pickServerId(@Nullable Integer initialServerId) {
        if (initialServerId == null) {
            return ThreadLocalRandom.current().nextInt(1 << 15) << 16;
        } else {
            return initialServerId;
        }
    }

    /**
     * try-with-resources класс для восстановления исходного каталога в javax.sql.Connection
     */
    public static class CatalogGuard implements AutoCloseable {
        private final Connection connection;
        private final String initialCatalog;

        public CatalogGuard(@Nonnull Connection connection) throws SQLException {
            this.connection = connection;
            this.initialCatalog = connection.getCatalog();
        }

        @Override
        public void close() throws SQLException {
            if (!initialCatalog.isEmpty()) {
                connection.setCatalog(initialCatalog);
            }
        }
    }
}
