package ru.yandex.direct.mysql.schema;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.mysql.MySQLDataType;

public class ColumnSchema {
    private final String name;
    private final String dataType;
    private final String columnType;
    private final String defaultValue;
    private final boolean isNullable;
    private final MySQLDataType mysqlDataType;
    private final boolean isUnsigned;

    @JsonCreator
    public ColumnSchema(@JsonProperty("name") String name, @JsonProperty("data_type") String dataType,
                        @JsonProperty("column_type") String columnType,
                        @JsonProperty("default_value") String defaultValue,
                        @JsonProperty("is_nullable") boolean isNullable) {
        this.name = Objects.requireNonNull(name);
        this.dataType = Objects.requireNonNull(dataType);
        this.columnType = Objects.requireNonNull(columnType);
        this.defaultValue = defaultValue;
        this.isNullable = isNullable;
        this.mysqlDataType = MySQLDataType.byName(dataType);
        this.isUnsigned = columnType.trim().toLowerCase().endsWith("unsigned");
    }

    @Override
    public String toString() {
        return "ColumnSchema{" +
                "name='" + name + '\'' +
                ", dataType='" + dataType + '\'' +
                ", columnType='" + columnType + '\'' +
                ", defaultValue='" + defaultValue + '\'' +
                ", isNullable=" + isNullable +
                '}';
    }

    @JsonGetter("name")
    public String getName() {
        return name;
    }

    @JsonGetter("data_type")
    public String getDataType() {
        return dataType;
    }

    public MySQLDataType mysqlDataType() {
        return mysqlDataType;
    }

    @JsonGetter("column_type")
    public String getColumnType() {
        return columnType;
    }

    @JsonGetter("default_value")
    public String getDefaultValue() {
        return defaultValue;
    }

    @JsonGetter("is_nullable")
    public boolean isNullable() {
        return isNullable;
    }

    @JsonIgnore
    public boolean isUnsigned() {
        return isUnsigned;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ColumnSchema)) {
            return false;
        }

        ColumnSchema that = (ColumnSchema) o;

        if (isNullable != that.isNullable) {
            return false;
        }
        if (!name.equals(that.name)) {
            return false;
        }
        if (!dataType.equals(that.dataType)) {
            return false;
        }
        if (!columnType.equals(that.columnType)) {
            return false;
        }
        return defaultValue != null ? defaultValue.equals(that.defaultValue) : that.defaultValue == null;

    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + dataType.hashCode();
        result = 31 * result + columnType.hashCode();
        result = 31 * result + (defaultValue != null ? defaultValue.hashCode() : 0);
        result = 31 * result + (isNullable ? 1 : 0);
        return result;
    }
}
