package ru.yandex.direct.mysql.schema;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.mysql.MySQLUtils;

public class RoutineSchema {
    private final String name;
    private final RoutineType type;
    private final String createSql;

    @JsonCreator
    public RoutineSchema(@JsonProperty("name") String name, @JsonProperty("type") RoutineType type,
                         @JsonProperty("create_sql") String createSql) {
        this.name = Objects.requireNonNull(name);
        this.type = Objects.requireNonNull(type);
        this.createSql = Objects.requireNonNull(createSql);
    }

    @JsonGetter("name")
    public String getName() {
        return name;
    }

    @JsonGetter("type")
    public RoutineType getType() {
        return type;
    }

    @JsonGetter("create_sql")
    public String getCreateSql() {
        return createSql;
    }

    public static RoutineSchema dump(Connection conn, String routineName, RoutineType routineType) throws SQLException {
        String createSql = null;
        try (PreparedStatement stmt = conn
                .prepareStatement("SHOW CREATE " + routineType + " " + MySQLUtils.quoteName(routineName))) {
            try (ResultSet rs = stmt.executeQuery()) {
                if (rs.next()) {
                    createSql = rs.getString(3);
                }
            }
        }
        if (createSql == null) {
            throw new IllegalStateException(
                    "Cannot get sql for CREATE " + routineType + " " + MySQLUtils.quoteName(routineName)
                            + " statement");
        }
        return new RoutineSchema(routineName, routineType, createSql);
    }

    public void restore(Connection conn) throws SQLException {
        MySQLUtils.executeUpdate(conn, createSql);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RoutineSchema)) {
            return false;
        }

        RoutineSchema that = (RoutineSchema) o;

        if (!name.equals(that.name)) {
            return false;
        }
        if (type != that.type) {
            return false;
        }
        return createSql.equals(that.createSql);

    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + type.hashCode();
        result = 31 * result + createSql.hashCode();
        return result;
    }
}
