package ru.yandex.direct.mysql.schema;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.mysql.MySQLUtils;
import ru.yandex.direct.utils.JsonUtils;

public class ServerSchema {
    // TODO: убрать adi после DIRECT-160170
    private static final Pattern ignoredDatabases =
            Pattern.compile("sys|mysql|information_schema|performance_schema|#mysql50#.*|xtrabackup_backupfiles|^adi$|^testdb$",
                    Pattern.CASE_INSENSITIVE | Pattern.DOTALL);

    private final List<DatabaseSchema> databases;

    public ServerSchema() {
        this.databases = new ArrayList<>();
    }

    @JsonCreator
    public ServerSchema(@JsonProperty("databases") List<DatabaseSchema> databases) {
        this.databases = Objects.requireNonNull(databases);
    }

    @JsonGetter("databases")
    public List<DatabaseSchema> getDatabases() {
        return databases;
    }

    public static ServerSchema dump(Connection conn) throws SQLException {
        MySQLUtils.executeUpdate(conn, "SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO'");
        MySQLUtils.executeUpdate(conn, "SET @OLD_TIME_ZONE=@@TIME_ZONE");
        MySQLUtils.executeUpdate(conn, "SET TIME_ZONE='+00:00'");
        List<String> databaseNames = new ArrayList<>();
        try (PreparedStatement stmt = conn.prepareStatement("SHOW DATABASES")) {
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    String databaseName = rs.getString(1);
                    if (isIgnoredDatabase(databaseName)) {
                        continue;
                    }
                    databaseNames.add(databaseName);
                }
            }
        }
        List<DatabaseSchema> databases = new ArrayList<>();
        for (String databaseName : databaseNames) {
            databases.add(DatabaseSchema.dump(conn, databaseName));
        }
        MySQLUtils.executeUpdate(conn, "SET TIME_ZONE=@OLD_TIME_ZONE");
        MySQLUtils.executeUpdate(conn, "SET SQL_MODE=@OLD_SQL_MODE");
        return new ServerSchema(databases);
    }

    public static boolean isIgnoredDatabase(String databaseName) {
        return ignoredDatabases.matcher(databaseName).matches();
    }

    public void restore(Connection conn) throws SQLException {
        MySQLUtils.executeUpdate(conn, "SET @OLD_TIME_ZONE=@@TIME_ZONE");
        MySQLUtils.executeUpdate(conn, "SET TIME_ZONE='+00:00'");
        MySQLUtils.executeUpdate(conn, "SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0");
        MySQLUtils.executeUpdate(conn, "SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0");
        MySQLUtils.executeUpdate(conn, "SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0");
        MySQLUtils.executeUpdate(conn, "SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO'");
        for (DatabaseSchema database : databases) {
            if (isIgnoredDatabase(database.getName())) {
                continue;
            }
            database.restore(conn);
        }
        for (DatabaseSchema database : databases) {
            if (isIgnoredDatabase(database.getName())) {
                continue;
            }
            database.restoreViews(conn);
        }
        MySQLUtils.executeUpdate(conn, "SET SQL_MODE=@OLD_SQL_MODE");
        MySQLUtils.executeUpdate(conn, "SET FOREIGN_KEY_CHECKS=@OLD_FOREIGN_KEY_CHECKS");
        MySQLUtils.executeUpdate(conn, "SET UNIQUE_CHECKS=@OLD_UNIQUE_CHECKS");
        MySQLUtils.executeUpdate(conn, "SET SQL_NOTES=@OLD_SQL_NOTES");
        MySQLUtils.executeUpdate(conn, "SET TIME_ZONE=@OLD_TIME_ZONE");
    }

    private static final ObjectMapper objectMapper = new ObjectMapper();

    public String toJson() throws JsonProcessingException {
        return objectMapper.writeValueAsString(this);
    }

    public byte[] toJsonBytes() {
        return JsonUtils.toJsonBytes(this);
    }

    public static ServerSchema fromJson(byte[] json) {
        if (Objects.isNull(json)) {
            return null;
        }
        return JsonUtils.fromJson(json, ServerSchema.class);
    }

    public boolean schemaEquals(ServerSchema that) {
        if (databases.size() != that.databases.size()) {
            return false;
        }
        for (int i = 0; i < databases.size(); i++) {
            if (!databases.get(i).schemaEquals(that.databases.get(i))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ServerSchema)) {
            return false;
        }

        ServerSchema that = (ServerSchema) o;

        return databases.equals(that.databases);

    }

    @Override
    public int hashCode() {
        return databases.hashCode();
    }
}
